
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

use work.nacl_constants.all;


-----------------------------------------
-- paddr:                              --
--  00 .. write: start[pwdata]         --
--        read:  ..000 & done & error  --
--  01 .. read:  needed cycles         --
--  10 .. write: RAM[address] = pwdata --
--  11 .. write:     address  = pwdata --
--        read:  RAM[address]          --
-----------------------------------------
entity nacl is
  port (
    pclk    :  in std_logic;
    presetn :  in std_logic;
    paddr   :  in std_logic_vector(1 downto 0);
    psel    :  in std_logic;
    penable :  in std_logic;
    pwrite  :  in std_logic;
    pwdata  :  in std_logic_vector(WORD_SIZE-1 downto 0);
    prdata  : out std_logic_vector(WORD_SIZE-1 downto 0)
  );
end nacl;

architecture behaviour of nacl is

  signal clk             : std_logic;
  signal rst             : std_logic;
  signal start           : std_logic;
  signal done            : std_logic;
  signal error           : std_logic;
  signal address         : std_logic_vector(ADDRESS_WIDTH-1 downto 0);
  signal address_buf     : std_logic_vector(PUBLIC_ADDRESS_WIDTH-1 downto 0);
  signal address_en      : std_logic;
  signal data_wr         : std_logic;
  signal data_out        : std_logic_vector(WORD_SIZE-1 downto 0);
  signal cycle_counter   : std_logic_vector(WORD_SIZE-1 downto 0);
  signal start_point_sel : std_logic_vector(START_POINTS_LD-1 downto 0);
  signal start_point_buf : std_logic_vector(START_POINTS_LD-1 downto 0);
  signal start_point_en  : std_logic;

begin

  core : entity work.nacl_core
    port map (
      clk             => clk,
      rst             => rst,
      start           => start,
      done            => done,
      error           => error,
      address         => address,
      data_wr         => data_wr,
      data_in         => pwdata,
      data_out        => data_out,
      cycle_counter   => cycle_counter,
      sel_start_point => start_point_sel
    );

  startPointBuffer : entity work.nacl_register
    generic map(
      WIDTH => START_POINTS_LD
    )
    port map(
      clk  => clk,
      en   => start_point_en,
      rst  => rst,
      din  => pwdata(START_POINTS_LD-1 downto 0),
      dout => start_point_buf
    );

  addressRegister : entity work.nacl_register
    generic map(
      WIDTH => PUBLIC_ADDRESS_WIDTH
    )
    port map(
      clk  => clk,
      en   => address_en,
      rst  => rst,
      din  => pwdata(PUBLIC_ADDRESS_WIDTH-1 downto 0),
      dout => address_buf
    );
  address(PUBLIC_ADDRESS_WIDTH-1 downto 0) <= address_buf;
  address(ADDRESS_WIDTH-1 downto PUBLIC_ADDRESS_WIDTH) <= (others => '0');

  clk <= pclk;
  rst <= not presetn;

  startControl : process(paddr, pwrite) begin
    if paddr = "00" and pwrite = '1' then
      start <= '1';
    else
      start <= '0';
    end if;
  end process;

  writeControl : process(paddr, pwdata, pwrite) begin
    if paddr = "10" and pwrite = '1' then
      data_wr <= '1';
    else
      data_wr <= '0';
    end if;
  end process;

  readControl : process(paddr, done, data_out, cycle_counter, error) begin
    if paddr = "11" then
      address_en <= '1';
      prdata <= data_out;
    elsif paddr = "01" then
      address_en <= '0';
      prdata <= cycle_counter;
    else
      address_en <= '0';
      prdata( 1 downto 0) <= done & error;
      prdata(31 downto 2) <= (others => '0');
    end if;
  end process;

  startPointMux : process(paddr, pwrite, pwdata(START_POINTS_LD-1 downto 0)) begin
    if paddr = "00" and pwrite = '1' then
      start_point_en  <= '1';
      start_point_sel <= pwdata(START_POINTS_LD-1 downto 0);
    else
      start_point_en  <= '0';
      start_point_sel <= start_point_buf;
    end if;
  end process;

end;