
library ieee;       
use ieee.numeric_std.all;                                                  
use ieee.std_logic_1164.all;                                          
use ieee.std_logic_unsigned.all;  

use work.nacl_constants.all;

entity nacl_alu is 
  port(
    clk              :  in std_logic;
    rst              :  in std_logic;    
    data_in          :  in std_logic_vector(WORD_SIZE-1 downto 0);   
    en_adder_ext     :  in std_logic_vector(NUM_ADDERS downto 0);
    en_carry         :  in std_logic;
    en_carry_wr      :  in std_logic;
    en_carry_inv     :  in std_logic;
    en_carry_xor     :  in std_logic;
    en_logic_xor     :  in std_logic;
    en_logic_or      :  in std_logic;
    en_logic_adder   :  in std_logic;
    en_subtract      :  in std_logic;
    en_registerA0      :  in std_logic;
    en_registerA1      :  in std_logic;
    en_registerB1      :  in std_logic;
    en_accu          :  in std_logic;
    rst_accu         :  in std_logic;
    rst_accu_low     :  in std_logic;
    rst_registerA      :  in std_logic;
    set_cin          :  in std_logic;
    sel_en_mode      :  in std_logic; 
    sel_add_mode     :  in std_logic; 
    sel_out_mux      :  in std_logic;
    sel_accu_mask    :  in std_logic_vector(1 downto 0);
    sel_rotation     :  in std_logic_vector(ROTATE_VARIANTS_LD-1 downto 0);  
    mult_counter     :  in std_logic_vector(MULT_CYCLES_LD-1 downto 0); 
    lc_index         :  in std_logic_vector(4 downto 0);
    lc_bit           : out std_logic;
    result           : out std_logic_vector(WORD_SIZE-1 downto 0);
    carry            : out std_logic
  );                              
end nacl_alu;                                                          

architecture behaviour of nacl_alu is
     
  signal cin             : std_logic;
  signal carry_in        : std_logic;
  signal carry_buffer    : std_logic;    
  signal accu_outE       : std_logic_vector(WORD_SIZE-1 downto 0);
  signal accu_out0       : std_logic_vector(WORD_SIZE-1 downto 0);
  signal accu_out1       : std_logic_vector(WORD_SIZE+2 downto 0);
  signal accu_out        : std_logic_vector(ACCU_SIZE+ACCU_EXTENSION-1 downto 0);
  signal result_int      : std_logic_vector(ACCU_SIZE+ACCU_EXTENSION-1 downto 0);
  signal accu_rot        : std_logic_vector(ACCU_SIZE+ACCU_EXTENSION-1 downto 0);
  signal rst_accu0       : std_logic;
  signal rst_accu1       : std_logic; 
  signal inA             : std_logic_vector(WORD_SIZE-1 downto 0);   
  signal inB             : std_logic_vector(WORD_SIZE-1 downto 0);
  signal inC             : std_logic_vector(ACCU_SIZE+ACCU_EXTENSION-1 downto 0);    
  signal bufferB_out     : std_logic_vector(WORD_SIZE-1 downto 0);  
  signal accu_mask       : std_logic_vector(ACCU_SIZE+ACCU_EXTENSION-1 downto 0);    
  signal accu_mask1      : std_logic_vector(ACCU_SIZE+ACCU_EXTENSION-1 downto 0);
  signal accu_mask3      : std_logic_vector(ACCU_SIZE+WORD_SIZE-1 downto 0);  

begin
 
  inputRegister: entity work.nacl_alu_register
    port map(
      clk           => clk, 
      rst           => rst,
      en_registerA0 => en_registerA0,
      en_registerA1 => en_registerA1,
      en_registerB1 => en_registerB1,
      rst_registerA => rst_registerA,
      data_in       => data_in,
      data_A        => inA,
      data_B        => bufferB_out
    );
 
  ALUCore : entity work.nacl_alu_core
    port map (
      inA            => inA,
      inB            => inB,
      inC            => inC,
      cin            => cin,
      en_adder_ext   => en_adder_ext,
      en_logic_xor   => en_logic_xor,
      en_logic_or    => en_logic_or,
      en_logic_adder => en_logic_adder,
      sel_en_mode    => sel_en_mode,     
      sel_add_mode   => sel_add_mode,
      mult_counter   => mult_counter,
      result         => result_int
    );
  inC <= accu_mask and accu_rot;
    
  accuExtra : if MULT_CYCLES = 2 and USE_MUL256 = 1 generate
    accuE : entity work.nacl_register
      generic map (
        WIDTH => WORD_SIZE
      )
      port map (
        clk  => clk,
        rst  => rst_accu1,
        en   => en_accu,
        din  => result_int(WORD_SIZE-1 downto 0),
        dout => accu_outE      
      );  
  end generate; 
  
  accu0 : entity work.nacl_register
    generic map (
      WIDTH => WORD_SIZE
    )
    port map (
      clk  => clk,
      rst  => rst_accu0,
      en   => en_accu,
      din  => result_int(WORD_SIZE+ACCU_EXTENSION-1 downto ACCU_EXTENSION),
      dout => accu_out0      
    );    
  rst_accu0 <= rst_accu1 or rst_accu_low;
    
  accu1 : entity work.nacl_register
    generic map (
      WIDTH => WORD_SIZE+3
    )
    port map (
      clk  => clk,
      rst  => rst_accu1,
      en   => en_accu,
      din  => result_int(ACCU_SIZE+ACCU_EXTENSION-1 downto WORD_SIZE+ACCU_EXTENSION),
      dout => accu_out1  
    ); 
  rst_accu1 <= rst or rst_accu;
    
  accu_mask1(  WORD_SIZE+ACCU_EXTENSION-NUM_ADDERS-2 downto 0) <= (others => '0');
  accu_mask1(2*WORD_SIZE+ACCU_EXTENSION-NUM_ADDERS+1 downto WORD_SIZE+ACCU_EXTENSION-NUM_ADDERS-1) <= "111" & X"FFFFFFFF";
  accu_mask1(ACCU_SIZE+ACCU_EXTENSION-1 downto 2*WORD_SIZE+ACCU_EXTENSION-NUM_ADDERS+2) <= (others => '0');
   
  accu_mask3(  WORD_SIZE-NUM_ADDERS-2 downto 0) <= (others => '0');
  accu_mask3(2*WORD_SIZE-NUM_ADDERS+1 downto WORD_SIZE-NUM_ADDERS-1) <= "111" & X"FFFFFFFF";
  accu_mask3(ACCU_SIZE+WORD_SIZE-1 downto 2*WORD_SIZE-NUM_ADDERS+2) <= (others => '0');
  
  muxSubtraction : process(bufferB_out, en_subtract) begin
    if en_subtract = '1' then
      inB <= not bufferB_out;
    else
      inB <= bufferB_out;
    end if;
  end process;  
  
  rotationLogic : entity work.nacl_alu_rotation_logic
    generic map (
      WIDTH => ACCU_SIZE+ACCU_EXTENSION
    )
    port map (    
      inp => accu_out,
      oup => accu_rot,
      sel => sel_rotation
    );  
  
  accuOut : process(accu_out0, accu_out1, accu_outE) begin 
    if MULT_CYCLES = 2 and USE_MUL256 = 1 then
      accu_out <= accu_out1 & accu_out0 & accu_outE;
    else
      accu_out <= accu_out1 & accu_out0;
    end if;
  end process;
  
  carryBuffer : entity work.nacl_secure_bit_buffer
    port map (
      clk  => clk,
      rst  => rst,
      din  => carry_in,
      dout => carry_buffer,
      en   => en_carry_wr
    );
    
  accuMaskMux3 : if MULT_CYCLES /= 2 or USE_MUL256 = 0 generate
    accuMaskMux : entity work.nacl_mux3
      generic map( 
        WIDTH => ACCU_SIZE+ACCU_EXTENSION
      )
      port map(
        in0  => (others => '1'),
        in1  => accu_mask1,
        in2  => (others => '0'),
        outp => accu_mask,
        sel  => sel_accu_mask
      );
  end generate;
  
  accuMaskMux2 : if MULT_CYCLES = 2 and USE_MUL256 = 1 generate
    accuMaskMux : entity work.nacl_mux4
      generic map( 
        WIDTH => ACCU_SIZE+ACCU_EXTENSION
      )
      port map(
        in0  => (others => '1'),
        in1  => accu_mask1,
        in2  => (others => '0'),
        in3  => accu_mask3,
        outp => accu_mask,
        sel  => sel_accu_mask
      );
  end generate;

  cinControl : process(en_carry, accu_out, set_cin, en_carry_inv, en_carry_xor, carry_buffer) begin
    if en_carry_inv = '1' then
      carry_in <= not accu_out(WORD_SIZE+ACCU_EXTENSION);
    elsif en_carry_xor = '1' then
      carry_in <= carry_buffer xor accu_out(WORD_SIZE+ACCU_EXTENSION);
    else
      carry_in <= accu_out(WORD_SIZE+ACCU_EXTENSION);
    end if;
    
    if set_cin = '1' then
      cin <= '1';
    else
      cin <= en_carry and accu_out(WORD_SIZE+ACCU_EXTENSION);
    end if;
  end process;

  resultMux : process(sel_out_mux, inC, result_int) begin
    if Mult_CYCLES = 2 and USE_MUL256 = 1 and sel_out_mux = '1' then
      result <= result_int(WORD_SIZE+ACCU_EXTENSION-1 downto ACCU_EXTENSION);    
    else
      result <= result_int(WORD_SIZE-1 downto 0);    
    end if;
  end process;
  
  lc_bit   <= bufferB_out(conv_integer(lc_index));
  carry    <= carry_buffer;
  
end;