
library ieee;       
use ieee.numeric_std.all;                                                  
use ieee.std_logic_1164.all;                                          
use ieee.std_logic_unsigned.all;  

use work.nacl_constants.all;

entity nacl_controller is 
  port(
    clk              :  in std_logic;
    rst              :  in std_logic;
    start            :  in std_logic;
    done             : out std_logic;
        
    en_adder_ext     : out std_logic_vector(NUM_ADDERS downto 0);
    en_carry         : out std_logic;
    en_carry_wr      : out std_logic;
    en_carry_inv     : out std_logic;
    en_carry_xor     : out std_logic;
    en_logic_xor     : out std_logic;
    en_logic_or      : out std_logic;
    en_logic_adder   : out std_logic;
    en_subtract      : out std_logic;
    switch_en        : out std_logic;
    switch_in        : out std_logic;
    en_bufALU        : out std_logic;
    en_registerA0    : out std_logic;
    en_registerA1    : out std_logic;
    en_registerB1    : out std_logic;    
    en_accu          : out std_logic;
    rst_accu         : out std_logic;
    rst_accu_low     : out std_logic;
    rst_registerA    : out std_logic;
    set_cin          : out std_logic;
    sel_en_mode      : out std_logic; 
    sel_add_mode     : out std_logic; 
    sel_out_mux      : out std_logic;
    sel_accu_mask    : out std_logic_vector(1 downto 0);
    sel_rotation     : out std_logic_vector(ROTATE_VARIANTS_LD-1 downto 0);
    sel_start_point  :  in std_logic_vector(START_POINTS_LD-1 downto 0);
    mult_counter     : out std_logic_vector(MULT_CYCLES_LD-1 downto 0);  
    lc_index         : out std_logic_vector(4 downto 0); 
    lc_bit           :  in std_logic;    
    carry            :  in std_logic;
    ram_we           : out std_logic;
    ram_address      : out std_logic_vector(ADDRESS_WIDTH-1 downto 0)    
  );                               
end nacl_controller;                                                          

architecture behaviour of nacl_controller is  
  
  signal code            : std_logic_vector(8 downto 0);
  signal code_rom        : std_logic_vector(8 downto 0);
  signal code_register     : std_logic_vector(8 downto 0);
  signal code_buffer_en  : std_logic;  
  signal code_register_rst : std_logic;  
  signal code_load       : std_logic_vector(8 downto 0);
  signal code_sel        : std_logic;
  signal sel_rom         : std_logic;

  signal parameter      : std_logic_vector(4 downto 0);  
    
  signal program_counter_rst   : std_logic;
  signal program_counter_en    : std_logic;
  signal program_counter       : std_logic_vector(PROGRAM_SIZE_LD-1 downto 0);  
  signal program_start         : std_logic_vector(PROGRAM_SIZE_LD-1 downto 0);  
  signal program_counter_inc   : std_logic_vector(PROGRAM_SIZE_LD-1 downto 0);  
 
  signal call_stack_push       : std_logic;
  signal call_stack_pop        : std_logic;
  
  signal page_index_inc     : std_logic_vector(BASE_ADDRESS_SIZE-1 downto 0);
  signal page_index_param   : std_logic_vector(ADDRESS_PAGE_SIZE_LD+5 downto 0);
  signal page_index_en      : std_logic;
  signal page_index_sel     : std_logic;  
  signal address_index_mult   : std_logic_vector(4 downto 0);  
  signal address_index_sel    : std_logic_vector(1 downto 0);
  signal sel_address_mux      : std_logic;
  
  signal loop_counter_en      : std_logic;
  signal loop_counter         : std_logic_vector(LOOP_COUNTER_BITS-1 downto 0);
  signal loop_counter_inc     : std_logic_vector(LOOP_COUNTER_BITS-1 downto 0);
  
  signal flag_mult_start      : std_logic;
  signal flag_mult256_start   : std_logic;
  signal flag_mult_done       : std_logic;
  
  signal en_registerA0_mult   : std_logic;
  signal en_registerA1_mult   : std_logic;
  signal en_registerB1_ctrl   : std_logic;
  signal en_registerB1_mult   : std_logic;
  signal en_adder_ctrl        : std_logic_vector(NUM_ADDERS downto 0);
  signal ram_we_ctrl          : std_logic;
  signal ram_we_mult          : std_logic;
  signal en_accu_ctrl         : std_logic;
  signal en_accu_mult         : std_logic;
  signal rst_accu_ctrl        : std_logic;
  signal rst_accu_mult        : std_logic;
  signal sel_en_mode_ctrl     : std_logic;
  signal sel_en_mode_mult     : std_logic;
  signal sel_rotation_ctrl    : std_logic_vector(ROTATE_VARIANTS_LD-1 downto 0);
  signal sel_rotation_mult    : std_logic_vector(ROTATE_VARIANTS_LD-1 downto 0);
  
  signal previous_lc_bit_in   : std_logic;
  signal previous_lc_bit      : std_logic;
  signal previous_lc_bit_en   : std_logic;
  
begin
 
  programCounter : entity work.nacl_controller_program_counter
    port map (
      clk                  => clk,
      rst                  => rst,
      call_stack_push      => call_stack_push,
      call_stack_pop       => call_stack_pop,
      call_target_index    => parameter(CALL_TARGETS_LD-1 downto 0),
      program_counter_rst  => program_counter_rst,
      program_counter_en   => program_counter_en,
      program_start        => program_start,
      program_counter_inc  => program_counter_inc,
      program_counter      => program_counter
    );
  program_counter_rst <= rst or start; 
  program_start <= start_point_rom(conv_integer(sel_start_point))(PROGRAM_SIZE_LD-1 downto 0);
 
  programROMs : entity work.nacl_controller_rom_program
    port map (
      program_counter => program_counter,
      code            => code_rom,
      sel_start_point => sel_start_point
    ); 
  
  codeBuffer : entity work.nacl_register
    generic map( 
      WIDTH => 9
    )
    port map(
      clk  => clk,
      en   => code_buffer_en,
      rst  => code_register_rst,
      din  => code_rom,
      dout => code_register
    );
  code_register_rst <= rst or start;
  
  codeMux : entity work.nacl_mux2
    generic map( 
      WIDTH => 9
    )
    port map(
      in0  => code_register,
      in1  => code_load,
      outp => code,
      sel  => code_sel
    );
  code_load <= I_NOP & "00000";
  parameter <= code(4 downto 0);
  
  instructionDecoder : entity work.nacl_controller_instruction_decoder
    port map (
      clk                 => clk,
      rst                 => rst,
      start               => start,
      done                => done,      
      call_stack_push     => call_stack_push,
      call_stack_pop      => call_stack_pop,
      code_buffer_en      => code_buffer_en,
      code_sel            => code_sel,
      code                => code,    
      page_index_sel      => page_index_sel,
      page_index_en       => page_index_en,
      page_index_inc      => page_index_inc,
      page_index_param    => page_index_param,
      loop_counter_en     => loop_counter_en,
      loop_counter_inc    => loop_counter_inc,
      loop_counter        => loop_counter,
      address_index_sel   => address_index_sel,
      sel_address_mux     => sel_address_mux,
      program_counter_en  => program_counter_en,
      program_counter_inc => program_counter_inc,        
      flag_mult_start     => flag_mult_start,
      flag_mult256_start  => flag_mult256_start,
      flag_mult_done      => flag_mult_done,      
      rst_accu_ctrl       => rst_accu_ctrl,
      rst_accu_low        => rst_accu_low,      
      en_adder_ctrl       => en_adder_ctrl,
      en_carry            => en_carry,
      en_carry_wr         => en_carry_wr,
      en_carry_inv        => en_carry_inv,
      en_carry_xor        => en_carry_xor,
      en_logic_xor        => en_logic_xor,
      en_logic_or         => en_logic_or,
      en_logic_adder      => en_logic_adder,
      en_subtract         => en_subtract,
      switch_en           => switch_en,
      switch_in           => switch_in,
      en_bufALU           => en_bufALU,
      en_registerB1_ctrl  => en_registerB1_ctrl,
      en_accu_ctrl        => en_accu_ctrl,
      set_cin             => set_cin,
      sel_en_mode_ctrl    => sel_en_mode_ctrl,
      sel_add_mode        => sel_add_mode,
      sel_rotation_ctrl   => sel_rotation_ctrl,
      sel_start_point     => sel_start_point,
      lc_bit              => lc_bit,
      previous_lc_bit_in  => previous_lc_bit_in,
      previous_lc_bit     => previous_lc_bit,
      previous_lc_bit_en  => previous_lc_bit_en,
      carry               => carry,
      ram_we_ctrl         => ram_we_ctrl
    );
  
  multiplicationController : entity work.nacl_controller_multiply
    port map (
      clk           => clk,
      rst           => rst,
      mult_start    => flag_mult_start,
      mult256_start => flag_mult256_start,
      mult_done     => flag_mult_done,
      en_registerA0 => en_registerA0_mult,
      en_registerA1 => en_registerA1_mult,
      en_registerB1 => en_registerB1_mult,
      en_accu       => en_accu_mult,
      rst_accu      => rst_accu_mult,
      sel_en_mode   => sel_en_mode_mult,
      sel_out_mux   => sel_out_mux,
      sel_accu_mask => sel_accu_mask,
      sel_rotation  => sel_rotation_mult,
      address_index => address_index_mult,
      mult_counter  => mult_counter,
      ram_we        => ram_we_mult
    );  
  
  memoryManagementUnit : entity work.nacl_controller_memory_management_unit
    port map (
      clk                 => clk,
      rst                 => rst,
      page_index_inc      => page_index_inc,
      page_index_param    => page_index_param,
      page_index_en       => page_index_en,
      page_index_sel      => page_index_sel,
      address_index_param => parameter,
      address_index_mult  => address_index_mult,
      address_index_sel   => address_index_sel,
      loop_counter_en     => loop_counter_en,
      loop_counter_inc    => loop_counter_inc,
      loop_counter        => loop_counter,
      sel_address_mux     => sel_address_mux,
      ram_address         => ram_address
    );
    
  previousLCBitBuffer : entity work.nacl_secure_bit_buffer
    generic map (
      initial => "1001"
    )
    port map (
      clk  => clk,
      rst  => rst,
      din  => previous_lc_bit_in,
      dout => previous_lc_bit,
      en   => previous_lc_bit_en
    );
  
  ctrlMux : process(rst, code, ram_we_ctrl, ram_we_mult, en_accu_ctrl, en_accu_mult, rst_accu_ctrl, rst_accu_mult,
                    en_registerA0_mult, en_registerA1_mult, en_registerB1_ctrl, en_registerB1_mult,
                    en_adder_ctrl, sel_en_mode_ctrl, sel_en_mode_mult, sel_rotation_ctrl, sel_rotation_mult) begin
    if USE_MUL256 = 1 and code = I_MUL256 then
      ram_we        <= ram_we_mult;
      en_registerA0 <= en_registerA0_mult;
      en_registerA1 <= en_registerA1_mult;
      en_registerB1 <= en_registerB1_mult;
      en_adder_ext  <= (others => '0');
      rst_registerA <= rst;
      en_accu       <= en_accu_mult;
      rst_accu      <= rst_accu_mult;
      sel_en_mode   <= sel_en_mode_mult;
    else
      ram_we        <= ram_we_ctrl;
      en_registerA0 <= '1';
      en_registerA1 <= '1';
      en_registerB1 <= en_registerB1_ctrl;
      en_adder_ext  <= en_adder_ctrl;
      rst_registerA <= '1';
      en_accu       <= en_accu_ctrl;
      rst_accu      <= rst_accu_ctrl;
      sel_en_mode   <= sel_en_mode_ctrl;
    end if;
    
    if USE_MUL256 = 1 and code = I_MUL256 then
      sel_rotation <= sel_rotation_mult;
    elsif code(8 downto 4) = I_MULT then
      sel_rotation <= sel_rotation_mult;
    else
      sel_rotation <= sel_rotation_ctrl;
    end if;
    
  end process;

  lc_index <= loop_counter(4 downto 0);
  
end;
