
library ieee;       
use ieee.numeric_std.all;                                                  
use ieee.std_logic_1164.all;                                          
use ieee.std_logic_unsigned.all;  

use work.nacl_constants.all;

entity nacl_controller_instruction_decoder is 
  port(
    clk                 :  in std_logic;
    rst                 :  in std_logic;
    start               :  in std_logic;
    done                : out std_logic;
    call_stack_push     : out std_logic;
    call_stack_pop      : out std_logic;
    code_buffer_en      : out std_logic; 
    code_sel            : out std_logic;
    code                :  in std_logic_vector(8 downto 0);
    page_index_sel      : out std_logic;  
    page_index_en       : out std_logic;
    page_index_inc      : out std_logic_vector(BASE_ADDRESS_SIZE-1 downto 0);
    page_index_param    : out std_logic_vector(ADDRESS_PAGE_SIZE_LD+5 downto 0);
    loop_counter_en     : out std_logic;
    loop_counter_inc    : out std_logic_vector(LOOP_COUNTER_BITS-1 downto 0);
    loop_counter        :  in std_logic_vector(LOOP_COUNTER_BITS-1 downto 0);
    address_index_sel   : out std_logic_vector(1 downto 0);
    sel_address_mux     : out std_logic;
    program_counter_en  : out std_logic;
    program_counter_inc : out std_logic_vector(PROGRAM_SIZE_LD-1 downto 0);
    flag_mult_start     : out std_logic;
    flag_mult256_start  : out std_logic;
    flag_mult_done      :  in std_logic;    
    rst_accu_ctrl       : out std_logic;
    rst_accu_low        : out std_logic;
    en_adder_ctrl       : out std_logic_vector(NUM_ADDERS downto 0);
    en_carry            : out std_logic;
    en_carry_wr         : out std_logic;
    en_carry_inv        : out std_logic;
    en_carry_xor        : out std_logic;
    en_logic_xor        : out std_logic;
    en_logic_or         : out std_logic;
    en_logic_adder      : out std_logic;
    en_subtract         : out std_logic;
    switch_en           : out std_logic;
    switch_in           : out std_logic;
    en_bufALU           : out std_logic;
    en_registerB1_ctrl  : out std_logic;    
    en_accu_ctrl        : out std_logic;
    set_cin             : out std_logic;
    sel_en_mode_ctrl    : out std_logic; 
    sel_add_mode        : out std_logic; 
    sel_rotation_ctrl   : out std_logic_vector(ROTATE_VARIANTS_LD-1 downto 0);
    sel_start_point     :  in std_logic_vector(START_POINTS_LD-1 downto 0);
    lc_bit              :  in std_logic;
    previous_lc_bit_in  : out std_logic;
    previous_lc_bit     :  in std_logic;
    previous_lc_bit_en  : out std_logic;
    carry               :  in std_logic;
    ram_we_ctrl         : out std_logic
  );                               
end nacl_controller_instruction_decoder;                                                          

architecture behaviour of nacl_controller_instruction_decoder is  
  
  constant page_index_offset : std_logic_vector(ADDRESS_PAGE_SIZE_LD-1 downto 0) := (others => '0');
  
  signal state_current               : controller_state_type;
  signal state_next                  : controller_state_type;
  
  signal instruction                 : std_logic_vector(3 downto 0);
  signal parameter                   : std_logic_vector(4 downto 0);  
  
  signal adder_constant_index        : std_logic_vector(ADDER_CONSTANTS_LD-1 downto 0);
  signal adder_constant_index_lim    : std_logic_vector(ADDER_CONSTANTS_LD-1 downto 0);
  
  signal external_adder_setting      : std_logic_vector(NUM_ADDERS downto 0);
  signal external_adder_setting_mask : std_logic_vector(NUM_ADDERS downto 0);

  signal flag_halt                   : std_logic;  
  signal flag_skip                   : std_logic;
  signal flag_mult_start_int         : std_logic;
  signal flag_mult256_start_int      : std_logic;
  
begin

  state : process (rst, clk) begin
    if rst = '1' then
      state_current <= S_WAIT;
    elsif rising_edge(clk) then
      state_current <= state_next;
    end if;
  end process;
  
  nextStateLogic : process(state_current, start, flag_halt, flag_mult_start_int, flag_mult256_start_int, flag_mult_done) begin
    case state_current is     
      when S_DO   => if    flag_halt = '1'              then state_next <= S_WAIT;
                     elsif flag_mult_start_int = '1'    then state_next <= S_MULT;
                     elsif flag_mult256_start_int = '1' then state_next <= S_MULT;
                     else                                    state_next <= S_DO;
                            end if;
      when S_MULT => if flag_mult_done = '1' then state_next <= S_DO;
                     else                         state_next <= S_MULT;
                     end if;
      when others => if start = '1' then state_next <= S_DO;
                     else                state_next <= S_WAIT;
                     end if;
     end case;
  end process;
  
  instructionDecoder: process(state_current, code, instruction, parameter, flag_mult_done, lc_bit, 
                    external_adder_setting_mask, previous_lc_bit, carry, 
                     external_adder_setting) begin    
    
    if state_current = S_DO then
      program_counter_en <= '1';
    elsif state_current = S_MULT then
      program_counter_en <= flag_mult_done;
    else
      program_counter_en <= '0';
    end if;
    
    if state_current = S_DO and code(8 downto 4) = I_MULT then
      code_buffer_en     <= '0';
    elsif state_current = S_DO and code = I_HLT then
      code_buffer_en     <= '0';
    elsif state_current = S_DO then
      code_buffer_en     <= '1';
    elsif state_current = S_MULT then
      code_buffer_en     <= flag_mult_done;
    else
      code_buffer_en     <= '0';
    end if;
    
    if state_current = S_MULT then
      program_counter_inc <= (0 => flag_mult_done, others => '0');
    elsif code(8 downto 4) = I_MULT or code = I_HLT then
      program_counter_inc <= (others => '0');     
    else
      program_counter_inc <= (0 => '1', others => '0');
    end if;
    
    if code = I_HLT then
      flag_halt <= '1';
    else
      flag_halt <= '0';
    end if;    
    
    if code = I_SW0 then
      previous_lc_bit_in <= '0';
    elsif code = I_SW1 then
      previous_lc_bit_in <= '1';
    else
      previous_lc_bit_in <= lc_bit;
    end if;
    
    if code = I_SWLC then
      previous_lc_bit_en <= '1';
    elsif code = I_SW0 or code = I_SW1 then
      previous_lc_bit_en <= '1';
    else
      previous_lc_bit_en <= '0';
    end if;
    
    if code = I_SW0 or code = I_SW1 then
      switch_en <= '1';
    elsif code = I_SWLC and (lc_bit xor previous_lc_bit) = '0' then
      switch_en <= '1';
    elsif code = I_SWLC and (lc_bit xor previous_lc_bit) = '1' then
      switch_en <= '1';
    else
      switch_en <= '0';
    end if;
    
    if code = I_SW1 then
      switch_in <= '1';
    elsif code = I_SWLC and (lc_bit xor previous_lc_bit) = '1' then
      switch_in <= '1';
    else
      switch_in <= '0';
    end if;    
        
    if instruction = I_ST or instruction = I_STR then
      ram_we_ctrl <= '1';
    else
      ram_we_ctrl <= '0';    
    end if;
    
    if instruction = I_CLR then
      rst_accu_ctrl <= '1';
    else
      rst_accu_ctrl <= '0';
    end if;
    
    if instruction = I_CLRL then
      rst_accu_low <= '1';
    else
      rst_accu_low <= '0';
    end if;
    
    if instruction = I_LD or instruction = I_LDC then
      en_registerB1_ctrl <= '1';
    else
      en_registerB1_ctrl <= '0';
    end if;
    
    if instruction = I_LDC then
      en_bufALU <= '1';
    else
      en_bufALU <= '0';
    end if;
    
    if code(8 downto 4) = I_MULT and code(1) = '0' then
      flag_mult_start_int <= '1';      
    else
      flag_mult_start_int <= '0';
    end if;
    
    if code(8 downto 4) = I_MULT and code(1) = '1' and state_current /= S_MULT then
      flag_mult256_start_int <= '1';
    else
      flag_mult256_start_int <= '0';
    end if;
    
    if code = I_OR then
      en_logic_or    <= '1';
      en_logic_xor   <= '0';
      en_logic_adder <= '0';
    elsif code = I_EOR then
      en_logic_or    <= '0';
      en_logic_xor   <= '1';
      en_logic_adder <= '0';
    elsif code = I_AND then
      en_logic_or    <= '0';
      en_logic_xor   <= '0';
      en_logic_adder <= '0';
    else
      en_logic_or    <= '0';
      en_logic_xor   <= '0';
      en_logic_adder <= '1';
    end if;
    
    if instruction = I_ROL32 then
      sel_rotation_ctrl <= (0 => '1', others => '0');
    elsif code(8 downto 4) = I_MULT then
      sel_rotation_ctrl <= (1 => '1', others => '0');    
    elsif instruction = I_STR then
      sel_rotation_ctrl <= (0 => '1', 1 => '1', others => '0');
    elsif code(8 downto 4) = I_ROT then
      sel_rotation_ctrl <= parameter(ROTATE_VARIANTS_LD-1 downto 0);
    else
      sel_rotation_ctrl <= (others => '0');
    end if;
    
    if instruction = I_ROL32 or code(8 downto 4) = I_ROT or code(8 downto 4) = I_MULT or 
       instruction = I_CALC or instruction = I_STR or
       (instruction = I_EXTRA and code(2) = '1' and code(8 downto 1) /= I_SFID) then
      en_accu_ctrl <= '1';   
    else
      en_accu_ctrl <= '0';   
    end if;
    
    if code(8 downto 2) = I_MULADDC or code(8 downto 2) = I_MULSUBC then
      en_carry <= '1';
    else
      en_carry <= '0';   
    end if;
    
    if code(8 downto 2) = I_MULADD  or code(8 downto 2) = I_MULSUB or 
       code(8 downto 2) = I_MULADDC or code(8 downto 2) = I_MULSUBC then
      sel_add_mode <= '1';  
    elsif code = I_AND or code = I_OR or code = I_EOR then
      sel_add_mode <= '1';        
    else
      sel_add_mode <= '0';
    end if;
    
    if code(8 downto 4) = I_MULT then
      sel_en_mode_ctrl <= '1';  
    else
      sel_en_mode_ctrl <= '0';
    end if;
    
    if code(8 downto 2) = I_MULSUB then
      set_cin <= '1';
    else
      set_cin <= '0';
    end if;   
    
    if code(8 downto 2) = I_MULSUB   or code(8 downto 2) = I_MULSUBC then    
      en_subtract <= '1';
    else
      en_subtract <= '0';
    end if;
    
    if code = I_STC then
      en_carry_wr  <= '1';   
      en_carry_inv <= '0';  
      en_carry_xor <= '0';
    elsif code = I_STI then
      en_carry_wr  <= '1';  
      en_carry_inv <= '1';  
      en_carry_xor <= '0';  
    elsif code = I_STX then
      en_carry_wr  <= '1';  
      en_carry_inv <= '0'; 
      en_carry_xor <= '1'; 
    else 
      en_carry_wr  <= '0';
      en_carry_inv <= '0';  
      en_carry_xor <= '0';
    end if;
    
    if code(8 downto 4) = I_ROT or instruction = I_CALC or 
       instruction(3 downto 1) = I_MPS or instruction = I_EXTRA or (code(8 downto 4) = I_MULT and code(1) = '0') then
      address_index_sel <= "00";
    elsif instruction = I_COND then
      address_index_sel <= "00";
    elsif state_current = S_WAIT then
      address_index_sel <= "00";    
    elsif code = I_MUL256 then
      address_index_sel <= "10";
    else
      address_index_sel <= "01";
    end if;
    
    if instruction = I_ST or instruction = I_ROL32 or code(8 downto 4) = I_ROT or instruction = I_STR then
      en_adder_ctrl <= (others => '0');
    elsif code = I_AND or code = I_OR or code = I_EOR then
      en_adder_ctrl <= (0 => '1', others => '0');
    elsif parameter(1 downto 0) = "00" then
      en_adder_ctrl <= external_adder_setting and external_adder_setting_mask;
    else
      en_adder_ctrl <= external_adder_setting;
    end if;
    
    if code = I_MPD then
      page_index_inc   <= (0 => '0', others => '1');
    else
      page_index_inc   <= (others => '0');
    end if;
    
    if code = I_MPI then
      page_index_en    <= '1';
      page_index_sel   <= '1';
    elsif code = I_MPD then
      page_index_en    <= '1';
      page_index_sel   <= '1';
    elsif instruction(3 downto 1) = I_MPS then
      page_index_en    <= '1';
      page_index_sel   <= '0'; 
    else
      page_index_en    <= '0';
      page_index_sel   <= '0';
    end if;    
    
    if ADDRESS_PAGE_SIZE > 1 then
      page_index_param <= instruction(0) & parameter & page_index_offset(ADDRESS_PAGE_SIZE_LD-1 downto 0);
    else
      page_index_param <= '0' & instruction(0) & parameter;
    end if;
    
    if code(8 downto 4) = I_JMP then
      call_stack_push     <= '1';
      call_stack_pop      <= '0';        
    elsif code = I_RET then
      call_stack_push     <= '0';
      call_stack_pop      <= '1';
    else
      call_stack_push     <= '0'; 
      call_stack_pop      <= '0';      
    end if;    
    
    if state_current = S_WAIT then
      done <= '1';
    else
      done <= '0';
    end if; 
    
    if instruction = I_FLC then
      sel_address_mux <= '1';
    else
      sel_address_mux <= '0';
    end if;   

    if code(8 downto 2) = I_SLCI or code(8 downto 2) = I_SLCD or code = I_INCLC or code = I_DECLC then
      loop_counter_en <= '1';
    else
      loop_counter_en <= '0';
    end if;
    
    if code(8 downto 2) = I_SLCD or code = I_DECLC then
      loop_counter_inc <= (others => '1');
    else
      loop_counter_inc <= (0 => '1', others => '0');
    end if;    
    
  end process;

  skipComparator : process(loop_counter, code(SKIP_CONSTANTS_LD-1 downto 0), code, lc_bit) begin
    if (code(8 downto 2) = I_SLCI or code(8 downto 2) = I_SLCD) and
      loop_counter = skip_constant_rom(conv_integer(code(SKIP_CONSTANTS_LD-1 downto 0))) then
      flag_skip <= '1';
    elsif code(8 downto 1) = I_SFID then
      if code(0) = '0' and sel_start_point = start_point_constant_rom(0) then
        flag_skip <= '1';
      elsif code(0) = '1' and sel_start_point = start_point_constant_rom(1) then
        flag_skip <= '1';
      else
        flag_skip <= '0';
      end if; 
    else
      flag_skip <= '0';
    end if;
  end process;
  
  codeSel : process(rst, clk) begin
    if rst = '1' then
      code_sel <= '0';
    elsif rising_edge(clk) then
      if code(8 downto 4) = I_JMP then
        code_sel <= '1';    
      elsif code = I_RET then
        code_sel <= '1';
      elsif (code(8 downto 2) = I_SLCI or code(8 downto 2) = I_SLCD) and flag_skip = '1' then
        code_sel <= '1';
      elsif code(8 downto 1) = I_SFID and flag_skip = '1' then
        code_sel <= '1';
      else
        code_sel <= '0';
      end if;
    end if;
  end process;

  external_adder_setting <= external_adder_settings_rom(conv_integer(adder_constant_index_lim));  

  external_adder_setting_mask <= (others => carry);
  
  limiterAdderSetting : entity work.nacl_address_limitation
    generic map(
      LIMIT         => ADDER_CONSTANTS,
      ADDRESS_WIDTH => ADDER_CONSTANTS_LD
    )
    port map(
      address_in  => adder_constant_index,
      address_out => adder_constant_index_lim
    );
  adder_constant_index <= parameter(ADDER_CONSTANTS_LD-1 downto 0);
  
  parameter   <= code(4 downto 0);
  instruction <= code(8 downto 5);

  flag_mult_start  <= flag_mult_start_int;
  flag_mult256_start <=  flag_mult256_start_int;
  
end;
