
library ieee;
use ieee.numeric_std.all;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

use work.nacl_constants.all;

entity nacl_controller_multiply is
  port(
    clk            :  in std_logic;
    rst            :  in std_logic;
    mult_start     :  in std_logic;
    mult256_start  :  in std_logic;
    mult_done      : out std_logic;
    en_registerA0  : out std_logic;
    en_registerA1  : out std_logic;
    en_registerB1  : out std_logic;
    en_accu        : out std_logic;
    rst_accu       : out std_logic;
    sel_en_mode    : out std_logic;
    sel_out_mux    : out std_logic;
    sel_accu_mask  : out std_logic_vector(1 downto 0);
    sel_rotation   : out std_logic_vector(ROTATE_VARIANTS_LD-1 downto 0);
    address_index  : out std_logic_vector(4 downto 0);
    mult_counter   : out std_logic_vector(MULT_CYCLES_LD-1 downto 0);
    ram_we         : out std_logic
  );
end nacl_controller_multiply;

architecture behaviour of nacl_controller_multiply is

  type state_type is (S_LOAD0, S_LOAD1, S_LOAD2, S_MULT, S_WRITE0, S_WRITE1, S_WRITE2);
  signal state_current : state_type;
  signal state_next    : state_type;

  type adress_rom_type is array(0 to 63) of std_logic_vector (10 downto 0);
  constant adress_rom : adress_rom_type := (
    '1' & "000" & "000" & "1110",

    '1' & "001" & "000" & "0000",
    '0' & "000" & "001" & "----",

    '1' & "010" & "000" & "0001",
    '0' & "001" & "001" & "----",
    '0' & "000" & "010" & "----",

    '1' & "011" & "000" & "0010",
    '0' & "010" & "001" & "----",
    '0' & "001" & "010" & "----",
    '0' & "000" & "011" & "----",

    '1' & "100" & "000" & "0011",
    '0' & "011" & "001" & "----",
    '0' & "010" & "010" & "----",
    '0' & "001" & "011" & "----",
    '0' & "000" & "100" & "----",

    '1' & "101" & "000" & "0100",
    '0' & "100" & "001" & "----",
    '0' & "011" & "010" & "----",
    '0' & "010" & "011" & "----",
    '0' & "001" & "100" & "----",
    '0' & "000" & "101" & "----",

    '1' & "110" & "000" & "0101",
    '0' & "101" & "001" & "----",
    '0' & "100" & "010" & "----",
    '0' & "011" & "011" & "----",
    '0' & "010" & "100" & "----",
    '0' & "001" & "101" & "----",
    '0' & "000" & "110" & "----",

    '1' & "111" & "000" & "0110",
    '0' & "110" & "001" & "----",
    '0' & "101" & "010" & "----",
    '0' & "100" & "011" & "----",
    '0' & "011" & "100" & "----",
    '0' & "010" & "101" & "----",
    '0' & "001" & "110" & "----",
    '0' & "000" & "111" & "----",

    '1' & "111" & "001" & "0111",
    '0' & "110" & "010" & "----",
    '0' & "101" & "011" & "----",
    '0' & "100" & "100" & "----",
    '0' & "011" & "101" & "----",
    '0' & "010" & "110" & "----",
    '0' & "001" & "111" & "----",

    '1' & "111" & "010" & "1000",
    '0' & "110" & "011" & "----",
    '0' & "101" & "100" & "----",
    '0' & "100" & "101" & "----",
    '0' & "011" & "110" & "----",
    '0' & "010" & "111" & "----",

    '1' & "111" & "011" & "1001",
    '0' & "110" & "100" & "----",
    '0' & "101" & "101" & "----",
    '0' & "100" & "110" & "----",
    '0' & "011" & "111" & "----",

    '1' & "111" & "100" & "1010",
    '0' & "110" & "101" & "----",
    '0' & "101" & "110" & "----",
    '0' & "100" & "111" & "----",

    '1' & "111" & "101" & "1011",
    '0' & "110" & "110" & "----",
    '0' & "101" & "111" & "----",

    '1' & "111" & "110" & "1100",
    '0' & "110" & "111" & "----",

    '1' & "111" & "111" & "1101"
  );

  type adress_rom_type2 is array(0 to 63) of std_logic_vector (9 downto 0);
  constant adress_rom2 : adress_rom_type2 := (
    '0' & '1' & '1' & "000" & "1101",
    '0' & '0' & '0' & "000" & "----",
    '0' & '1' & '0' & "001" & "----",

    '1' & '1' & '1' & "011" & "0000",
    '1' & '1' & '0' & "010" & "0001",
    '0' & '0' & '0' & "001" & "----",
    '1' & '1' & '0' & "001" & "----",
    '0' & '0' & '0' & "010" & "----",
    '1' & '1' & '0' & "000" & "----",
    '0' & '0' & '0' & "011" & "----",

    '1' & '0' & '1' & "101" & "0010",
    '1' & '0' & '0' & "100" & "0011",
    '0' & '1' & '0' & "001" & "----",
    '1' & '0' & '0' & "011" & "----",
    '0' & '1' & '0' & "010" & "----",
    '1' & '0' & '0' & "010" & "----",
    '0' & '1' & '0' & "011" & "----",
    '1' & '0' & '0' & "001" & "----",
    '0' & '1' & '0' & "100" & "----",
    '1' & '0' & '0' & "000" & "----",
    '0' & '1' & '0' & "101" & "----",

    '1' & '1' & '1' & "111" & "0100",
    '1' & '1' & '0' & "110" & "0101",
    '0' & '0' & '0' & "001" & "----",
    '1' & '1' & '0' & "101" & "----",
    '0' & '0' & '0' & "010" & "----",
    '1' & '1' & '0' & "100" & "----",
    '0' & '0' & '0' & "011" & "----",
    '1' & '1' & '0' & "011" & "----",
    '0' & '0' & '0' & "100" & "----",
    '1' & '1' & '0' & "010" & "----",
    '0' & '0' & '0' & "101" & "----",
    '1' & '1' & '0' & "001" & "----",
    '0' & '0' & '0' & "110" & "----",
    '1' & '1' & '0' & "000" & "----",
    '0' & '0' & '0' & "111" & "----",

    '1' & '1' & '1' & "001" & "0110",
    '0' & '1' & '0' & "010" & "0111",
    '1' & '0' & '0' & "110" & "----",
    '0' & '1' & '0' & "011" & "----",
    '1' & '0' & '0' & "101" & "----",
    '0' & '1' & '0' & "100" & "----",
    '1' & '0' & '0' & "100" & "----",
    '0' & '1' & '0' & "101" & "----",
    '1' & '0' & '0' & "011" & "----",
    '0' & '1' & '0' & "110" & "----",
    '1' & '0' & '0' & "010" & "----",
    '0' & '1' & '0' & "111" & "----",
    '1' & '0' & '0' & "001" & "----",

    '0' & '0' & '1' & "011" & "1000",
    '0' & '0' & '0' & "100" & "1001",
    '1' & '1' & '0' & "110" & "----",
    '0' & '0' & '0' & "101" & "----",
    '1' & '1' & '0' & "101" & "----",
    '0' & '0' & '0' & "110" & "----",
    '1' & '1' & '0' & "100" & "----",
    '0' & '0' & '0' & "111" & "----",
    '1' & '1' & '0' & "011" & "----",

    '0' & '1' & '1' & "101" & "1010",
    '0' & '1' & '0' & "110" & "1011",
    '1' & '0' & '0' & "110" & "----",
    '0' & '1' & '0' & "111" & "----",
    '1' & '0' & '0' & "101" & "----",

    '0' & '0' & '1' & "111" & "1100"
  );

  signal mult_counter_int : std_logic_vector(MULT_CYCLES_LD-1 downto 0);
  signal mult_counter_en  : std_logic;
  signal mult_counter_of  : std_logic;

  signal loop_counter     : std_logic_vector(5 downto 0);
  signal loop_counter_en  : std_logic;
  signal loop_counter_of  : std_logic;

  signal flag_done        : std_logic;
  signal flag_write       : std_logic;
  signal flag_write_del   : std_logic;
  signal flag_select      : std_logic;
  signal flag_shift       : std_logic;

  signal addr0            : std_logic_vector(2 downto 0);
  signal addr1            : std_logic_vector(2 downto 0);
  signal addr2            : std_logic_vector(3 downto 0);

  signal addrMux_in0      : std_logic_vector(4 downto 0);
  signal addrMux_in1      : std_logic_vector(4 downto 0);
  signal addrMux_in2      : std_logic_vector(4 downto 0);
  signal addrMux_outp     : std_logic_vector(4 downto 0);
  signal addrMux_sel      : std_logic_vector(1 downto 0);

begin

  multCounter0_3 : if MULT_CYCLES = 3 generate
    multCounter0 : entity work.nacl_count_to_3
      port map (
        clk      => clk,
        en       => mult_counter_en,
        rst      => rst,
        counter  => mult_counter_int,
        overflow => mult_counter_of
      );
  end generate;

  multCounter0 : if MULT_CYCLES /= 3 generate
    multCounter0 : entity work.nacl_counter
      generic map (
        WIDTH => MULT_CYCLES_LD
      )
      port map (
        clk      => clk,
        en       => mult_counter_en,
        rst      => rst,
        counter  => mult_counter_int,
        overflow => mult_counter_of
      );
  end generate;

  multCounter1 : entity work.nacl_counter
    generic map(
       WIDTH => 6
     )
    port map(
      clk      => clk,
      en       => loop_counter_en,
      rst      => rst,
      counter  => loop_counter,
      overflow => loop_counter_of
    );

  addresMux : entity work.nacl_mux3
    generic map(
       WIDTH => 5
     )
    port map(
      in0  => addrMux_in0,
      in1  => addrMux_in1,
      in2  => addrMux_in2,
      outp => addrMux_outp,
      sel  => addrMux_sel
     );

  state : process (clk, rst, state_next) begin
    if rst = '1' then
      state_current <= S_LOAD0;
    elsif rising_edge(clk) then
      state_current <= state_next;
    end if;
  end process;

  nsl : process(state_current, mult256_start, flag_done, mult_counter_int, flag_write) begin
    case state_current is
      when S_LOAD1 => state_next <= S_LOAD2;
      when S_LOAD2 => state_next <= S_MULT;
      when S_MULT  => if flag_done = '1' then                                                  state_next <= S_WRITE1;
                      elsif MULT_CYCLES = 2 and mult_counter_int = 1 and flag_write = '1' then state_next <= S_WRITE0;
                      else                                                                     state_next <= S_MULT;
                      end if;
      when S_WRITE0 => state_next <= S_MULT;
      when S_WRITE1 => state_next <= S_WRITE2;
      when S_WRITE2 => state_next <= S_LOAD0;
      when others  => if mult256_start = '1' then state_next <= S_LOAD1;
                      else                        state_next <= S_LOAD0;
                      end if;
     end case;
  end process;


  control : process(rst, state_current, mult_start, mult256_start, flag_select, flag_shift,
                    mult_counter_of, mult_counter_int, mult_counter_of, loop_counter, flag_write, flag_write_del) begin

    if state_current = S_LOAD1 then
      rst_accu <= '1';
    else
      rst_accu <= rst;
    end if;

    if mult_start = '1' then
      mult_counter_en <= '1';
    elsif state_current = S_MULT then
      mult_counter_en <= '1';
    else
      mult_counter_en <= '0';
    end if;

    if state_current = S_LOAD2 then
      loop_counter_en <= '1';
    elsif state_current = S_MULT and mult_counter_of = '1' and loop_counter /= 0 then
      loop_counter_en <= '1';
    else
      loop_counter_en <= '0';
    end if;

    if state_current = S_MULT and mult_counter_of = '1' and loop_counter = 0 then
      flag_done <= '1';
    else
      flag_done <= '0';
    end if;

    if state_current = S_LOAD0 then
      mult_done <= mult_counter_of;
    elsif state_current = S_WRITE2 then
      mult_done <= '1';
    else
      mult_done <= '0';
    end if;

    if state_current = S_LOAD1 then
      en_registerA0 <= '1';
    elsif MULT_CYCLES /=2 and state_current = S_MULT and mult_counter_int = 1 then
      en_registerA0 <= '1';
    elsif MULT_CYCLES = 2 and state_current = S_MULT and mult_counter_int = 1 and flag_select = '0' then
      en_registerA0 <= '1';
    else
      en_registerA0 <= '0';
    end if;

    if state_current = S_MULT and mult_counter_int = 0 then
      en_registerA1 <= '1';
    else
      en_registerA1 <= '0';
    end if;

    if MULT_CYCLES /= 2 and state_current = S_MULT and mult_counter_of = '1' then
      en_registerB1 <= '1';
    elsif MULT_CYCLES = 2 and state_current = S_MULT and mult_counter_int = 1 and flag_select = '1' then
      en_registerB1 <= '1';
    elsif state_current = S_LOAD2 then
      en_registerB1 <= '1';
    else
      en_registerB1 <= '0';
    end if;

    if state_current = S_MULT or state_current = S_WRITE1 then
      en_accu <= '1';
    else
      en_accu <= '0';
    end if;

    if state_current = S_MULT and mult_counter_of = '1' and flag_write = '1' then
      ram_we <= '1';
    elsif state_current = S_WRITE0 or state_current = S_WRITE2 then
      ram_we <= '1';
    else
      ram_we <= '0';
    end if;

    if MULT_CYCLES /= 2 and state_current = S_MULT and mult_counter_int = 0 and flag_write_del = '1' then
      sel_accu_mask <= "01";
      sel_rotation  <= (1 => '1', others => '0');
    elsif MULT_CYCLES = 2 and (loop_counter > 0 and loop_counter < 37) and state_current = S_MULT and mult_counter_int = 0 and flag_write_del = '1' then
      sel_accu_mask <= "11";
      sel_rotation  <= (2 => '1', 1 => '1', 0 => '1', others => '0');
    elsif MULT_CYCLES = 2 and loop_counter = 37 and state_current = S_MULT and mult_counter_int = 0 and flag_write_del = '1' then
      sel_accu_mask <= "01";
      sel_rotation  <= (1 => '1', others => '0');
    elsif MULT_CYCLES = 2 and state_current = S_MULT and mult_counter_int = 0 and flag_write_del = '1' then
      sel_accu_mask <= "01";
      sel_rotation  <= (2 => '1', 1 => '1', 0 => '1', others => '0');
    elsif MULT_CYCLES /= 2 and state_current = S_MULT and mult_counter_int = 0 and flag_write_del = '0' then
      sel_accu_mask <= "00";
      sel_rotation  <= (2 => '1', others => '0');
    elsif MULT_CYCLES = 2 and state_current = S_MULT and mult_counter_int = 0 and flag_write_del = '0' then
      sel_accu_mask <= "00";
      sel_rotation  <= (2 => not flag_shift, 1 => flag_shift, 0 => not flag_shift, others => '0');
    elsif state_current = S_WRITE0 then
      sel_accu_mask <= "00";
      sel_rotation  <= (2 => '1', 1 => '1', others => '0');
    elsif state_current = S_WRITE1 then
      sel_accu_mask <= "00";
      sel_rotation  <= (1 => '1', others => '0');
    elsif state_current = S_WRITE2 then
      sel_accu_mask <= "00";
      sel_rotation  <= (1 => '1', 0 => '1', others => '0');
    elsif MULT_CYCLES = 3 and mult_counter_int = 2 then
      sel_accu_mask <= "00";
      sel_rotation  <= (2 => '1', 0 => '1', others => '0');
    else
      sel_accu_mask <= "00";
      sel_rotation  <= (1 => '1', others => '0');
    end if;

    if state_current = S_LOAD0 then
      addrMux_sel <= "00";
    elsif state_current = S_LOAD1 or state_current = S_LOAD2 then
      addrMux_sel <= "01";
    elsif MULT_CYCLES /= 2 and state_current = S_MULT and mult_counter_int = 0 then
      addrMux_sel <= "00";
    elsif MULT_CYCLES = 2 and state_current = S_MULT and mult_counter_int = 0 and flag_select = '0' then
      addrMux_sel <= "00";
    elsif MULT_CYCLES = 2 and state_current = S_MULT and mult_counter_int = 0 and flag_select = '1' then
      addrMux_sel <= "01";
    elsif MULT_CYCLES /= 2 and mult_counter_of = '0' then
      addrMux_sel <= "01";
    else
      addrMux_sel <= "10";
    end if;

    if state_current = S_WRITE0 or state_current = S_WRITE1 or state_current = S_WRITE2 then
      sel_en_mode <= '0';
    else
      sel_en_mode <= '1';
    end if;

    if MULT_CYCLES = 2 and (loop_counter >= 38 or loop_counter = 0) then
      sel_out_mux <= '1';
    else
      sel_out_mux <= '0';
    end if;

  end process;

  addrMuxCtrl : process(state_current, addr0, addr1, addr2, addrMux_outp, loop_counter) begin
    if addr2(3) = '1' then
      addrMux_in2 <= "11" & addr2(2 downto 0);
    else
      addrMux_in2 <= "10" & addr2(2 downto 0);
    end if;

    if MULT_CYCLES = 2 then
      if state_current = S_LOAD0 then
        addrMux_in0 <= "00" & addr0(2 downto 1) & '1';
      else
        addrMux_in0 <= "00" & addr0;
      end if;
      addrMux_in1 <= "01" & addr0;
    else
      addrMux_in0 <= "00" & addr0;
      addrMux_in1 <= "01" & addr1;
    end if;

    if MULT_CYCLES = 2 and state_current = S_MULT and loop_counter = 0 then
      address_index <= "11110";
    elsif state_current = S_WRITE2 then
      address_index <= "11111";   
    else
      address_index <= addrMux_outp;
    end if;
  end process;

  flagWriteDel : process(rst, clk) begin
    if rst = '1' then
      flag_write_del <= '0';
    elsif rising_edge(clk) then
      if state_current = S_LOAD2 or (state_current = S_MULT and mult_counter_of = '1') then
        flag_write_del <= flag_write;
      end if;
    end if;
  end process;

  addressRom : process(loop_counter) begin
    if MULT_CYCLES = 2 then
      flag_shift   <= adress_rom2(conv_integer(loop_counter))(9);
      flag_select  <= adress_rom2(conv_integer(loop_counter))(8);
      flag_write   <= adress_rom2(conv_integer(loop_counter))(7);
      addr0        <= adress_rom2(conv_integer(loop_counter))(6 downto 4);
      addr2        <= adress_rom2(conv_integer(loop_counter))(3 downto 0);
    else
      flag_shift   <= '0';
      flag_select  <= '0';
      flag_write   <= adress_rom(conv_integer(loop_counter))(10);
      addr0        <= adress_rom(conv_integer(loop_counter))(9 downto 7);
      addr1        <= adress_rom(conv_integer(loop_counter))(6 downto 4);
      addr2        <= adress_rom(conv_integer(loop_counter))(3 downto 0);
    end if;
  end process;

  mult_counter <= mult_counter_int;

end;
