
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

use work.nacl_constants.all;

entity nacl_core is
  port (
    clk             :  in std_logic;
    rst             :  in std_logic;
    start           :  in std_logic;
    done            : out std_logic;
    error           : out std_logic;
    address         :  in std_logic_vector(ADDRESS_WIDTH-1 downto 0);
    data_wr         :  in std_logic;
    data_in         :  in std_logic_vector(WORD_SIZE-1 downto 0);
    data_out        : out std_logic_vector(WORD_SIZE-1 downto 0);
    cycle_counter   : out std_logic_vector(WORD_SIZE-1 downto 0);
    sel_start_point :  in std_logic_vector(START_POINTS_LD-1 downto 0)
  );
end nacl_core;

architecture behaviour of nacl_core is

  signal start_controller : std_logic;
  signal done_int         : std_logic;
  signal en_adder_ext     : std_logic_vector(NUM_ADDERS downto 0);
  signal en_carry         : std_logic;
  signal en_carry_inv     : std_logic;
  signal en_carry_wr      : std_logic;
  signal en_carry_xor     : std_logic;
  signal en_logic_xor     : std_logic;
  signal en_logic_or      : std_logic;
  signal en_logic_adder   : std_logic;
  signal en_subtract      : std_logic;
  signal switch_en          : std_logic;
  signal switch_in          : std_logic;
  signal en_bufALU        : std_logic;
  signal en_registerA0      : std_logic;
  signal en_registerA1      : std_logic;
  signal en_registerB1      : std_logic;
  signal en_accu          : std_logic;
  signal rst_accu         : std_logic;
  signal rst_accu_low     : std_logic;
  signal rst_registerA      : std_logic;
  signal set_cin          : std_logic;
  signal sel_en_mode      : std_logic;
  signal sel_add_mode     : std_logic;
  signal sel_out_mux      : std_logic;
  signal sel_accu_mask    : std_logic_vector(1 downto 0);
  signal sel_rotation     : std_logic_vector(ROTATE_VARIANTS_LD-1 downto 0);
  signal mult_counter     : std_logic_vector(MULT_CYCLES_LD-1 downto 0);
  signal lc_index         : std_logic_vector(4 downto 0);
  signal lc_bit           : std_logic;
  signal overflow         : std_logic_vector(2 downto 0);
  signal carry            : std_logic;

  signal ram_we           : std_logic;
  signal ram_we_ctrl      : std_logic;
  signal ram_we_test      : std_logic;
  signal ram_address      : std_logic_vector(ADDRESS_WIDTH-1 downto 0);
  signal ram_address_ctrl : std_logic_vector(ADDRESS_WIDTH-1 downto 0);
  signal ram_address_test : std_logic_vector(ADDRESS_WIDTH-1 downto 0);
  signal ram_din          : std_logic_vector(WORD_SIZE-1 downto 0);
  signal ram_din_alu      : std_logic_vector(WORD_SIZE-1 downto 0);
  signal ram_din_test     : std_logic_vector(WORD_SIZE-1 downto 0);
  signal ram_dout         : std_logic_vector(WORD_SIZE-1 downto 0);
  signal ram_mux_sel      : std_logic_vector(1 downto 0);

  signal cycle_counter_en : std_logic;

  signal start_point      :  std_logic_vector(START_POINTS_LD-1 downto 0);
  signal start_point_test :  std_logic_vector(START_POINTS_LD-1 downto 0);

  signal start_calc     : std_logic;
  signal start_calc_del : std_logic;
  signal test_done      : std_logic;
  signal test_error     : std_logic;
  signal test_active    : std_logic;

begin

  cycleCounter : entity work.nacl_counter
    generic map(
      WIDTH => WORD_SIZE
    )
    port map(
      clk      => clk,
      en       => cycle_counter_en,
      rst      => rst,
      counter  => cycle_counter,
      overflow => open
    );

  alu : entity work.nacl_alu
    port map (
      clk              => clk,
      rst              => rst,
      data_in          => ram_dout,
      en_adder_ext     => en_adder_ext,
      en_carry         => en_carry,
      en_carry_wr      => en_carry_wr,
      en_carry_inv     => en_carry_inv,
      en_carry_xor     => en_carry_xor,
      en_logic_xor     => en_logic_xor,
      en_logic_or      => en_logic_or,
      en_logic_adder   => en_logic_adder,
      en_subtract      => en_subtract,
      en_registerA0    => en_registerA0,
      en_registerA1    => en_registerA1,
      en_registerB1    => en_registerB1,
      en_accu          => en_accu,
      rst_accu         => rst_accu,
      rst_accu_low     => rst_accu_low,
      rst_registerA    => rst_registerA,
      set_cin          => set_cin,
      sel_en_mode      => sel_en_mode,
      sel_add_mode     => sel_add_mode,
      sel_out_mux      => sel_out_mux,
      sel_accu_mask    => sel_accu_mask,
      sel_rotation     => sel_rotation,
      mult_counter     => mult_counter,
      lc_index         => lc_index,
      lc_bit           => lc_bit,
      result           => ram_din_alu,
      carry            => carry
  );

  controller : entity work.nacl_controller
    port map (
      clk              => clk,
      rst              => rst,
      start            => start_controller,
      done             => done_int,
      en_adder_ext     => en_adder_ext,
      en_carry         => en_carry,
      en_carry_wr      => en_carry_wr,
      en_carry_inv     => en_carry_inv,
      en_carry_xor     => en_carry_xor,
      en_logic_xor     => en_logic_xor,
      en_logic_or      => en_logic_or,
      en_logic_adder   => en_logic_adder,
      en_subtract      => en_subtract,
      switch_en        => switch_en,
      switch_in        => switch_in,
      en_bufALU        => en_bufALU,
      en_registerA0    => en_registerA0,
      en_registerA1    => en_registerA1,
      en_registerB1    => en_registerB1,
      en_accu          => en_accu,
      rst_accu         => rst_accu,
      rst_accu_low     => rst_accu_low,
      rst_registerA    => rst_registerA,
      set_cin          => set_cin,
      sel_en_mode      => sel_en_mode,
      sel_add_mode     => sel_add_mode,
      sel_out_mux      => sel_out_mux,
      sel_accu_mask    => sel_accu_mask,
      sel_rotation     => sel_rotation,
      sel_start_point  => start_point,
      mult_counter     => mult_counter,
      lc_index         => lc_index,
      lc_bit           => lc_bit,
      carry            => carry,
      ram_we           => ram_we_ctrl,
      ram_address      => ram_address_ctrl
    );

  memory : entity work.nacl_memory
    port map (
      clk       => clk,
      rst       => rst,
      ram_we    => ram_we,
      switch_en => switch_en,
      switch_in => switch_in,
      en_bufALU => en_bufALU,
      address   => ram_address,
      data_in   => ram_din,
      data_out  => ram_dout
    );

  selftest : if USE_SELFTEST = 1 generate
    selftest : entity work.nacl_selftest
      port map(
        clk             => clk,
        rst             => rst,
        start_calc      => start_calc,
        start_test      => start,
        calc_done       => done_int,
        test_done       => test_done,
        ram_addr        => ram_address_test,
        ram_in          => ram_din_test,
        ram_we          => ram_we_test,
        ram_data        => ram_dout,
        active          => test_active,
        error           => test_error,
        sel_start_point => start_point_test
      );
  end generate;

  addressMux : entity work.nacl_mux3
    generic map(
      WIDTH => ADDRESS_WIDTH
    )
    port map(
      in0  => ram_address_ctrl,
      in1  => address,
      in2  => ram_address_test,
      outp => ram_address,
      sel  => ram_mux_sel
    );

  dataMux : entity work.nacl_mux3
    generic map(
      WIDTH => WORD_SIZE
    )
    port map(
      in0  => ram_din_alu,
      in1  => data_in,
      in2  => ram_din_test,
      outp => ram_din,
      sel  => ram_mux_sel
    );

  wrMux : process(ram_mux_sel, data_wr, ram_we_ctrl, ram_we_test) begin
    if ram_mux_sel = "01" then
      ram_we <= data_wr;
    elsif ram_mux_sel = "10" then
      ram_we <= ram_we_test;
    else
      ram_we <= ram_we_ctrl;
    end if;
  end process;

  selftestMux : process(start, start_calc, done_int, test_done, test_error, start_point_test) begin
    if USE_SELFTEST = 1 then
      start_controller <= start_calc;
      done             <= test_done;
      error            <= test_error;
      start_point      <= start_point_test;
    else
      start_controller <= start;
      done             <= done_int;
      error            <= '0';
      start_point      <= sel_start_point;
    end if;
  end process;

  delay : process(rst, clk) begin
    if rst = '1' then
      start_calc_del <= '0';
    elsif rising_edge(clk) then
      start_calc_del <= start_controller;
    end if;
  end process;

  cycleCounterControl : process(rst, clk) begin
    if rst = '1' then
      cycle_counter_en <= '0';
    elsif rising_edge(clk) then
      if start_calc_del = '1' then
        cycle_counter_en <= '1';
      elsif done_int = '1' then
        cycle_counter_en <= '0';
      end if;
    end if;
  end process;

  selMux : process(test_active, done_int) begin
    if USE_SELFTEST = 1 then
      ram_mux_sel <= test_active & ((not test_active) and done_int);
    else
      ram_mux_sel <= '0' & done_int;
    end if;
  end process;

  data_out <= ram_dout; 

end;
