
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;

use work.nacl_constants.all;

entity nacl_memory is
  port (
    clk       :  in std_logic;
    rst       :  in std_logic;
    ram_we    :  in std_logic;
    switch_en   :  in std_logic;
    switch_in   :  in std_logic;
    en_bufALU :  in std_logic;
    address   :  in std_logic_vector(ADDRESS_WIDTH-1 downto 0);
    data_in   :  in std_logic_vector(WORD_SIZE-1 downto 0);
    data_out  : out std_logic_vector(WORD_SIZE-1 downto 0)
  );
end nacl_memory;

architecture behaviour of nacl_memory is

  constant ADDRESS_ROM_OFFSET  : std_logic_vector(ADDRESS_WIDTH-4 downto 0) := not std_logic_vector(to_unsigned(NUM_REGISTERS, ADDRESS_WIDTH-3));

  signal address_ram_limited   : std_logic_vector(ADDRESS_WIDTH_RAM-1 downto 0);
  signal address_ram           : std_logic_vector(ADDRESS_WIDTH_RAM-1 downto 0);    
  
  signal switch_state          : std_logic;
  signal switch_next           : std_logic;

  signal address_rom_int       : std_logic_vector(ADDRESS_WIDTH_ROM-1 downto 0);
  signal address_rom_limited   : std_logic_vector(ADDRESS_WIDTH_ROM-1 downto 0);
  signal address_rom_buffer_en : std_logic;
  signal address_rom_buffer    : std_logic_vector(ADDRESS_WIDTH_ROM-1 downto 0);
  signal address_rom           : std_logic_vector(ADDRESS_WIDTH_ROM-1 downto 0);
  
  signal output_sel            : std_logic;
  signal data_ram              : std_logic_vector(WORD_SIZE-1 downto 0);
  signal data_rom              : std_logic_vector(WORD_SIZE-1 downto 0);

begin

  ------ RAM --------------------------------------------------------

  limiterRAM : entity work.nacl_address_limitation
    generic map(
      LIMIT         => NUM_REGISTERS,
      ADDRESS_WIDTH => ADDRESS_WIDTH_RAM-3
    )
    port map(
      address_in  => address(ADDRESS_WIDTH_RAM-1 downto 3),
      address_out => address_ram_limited(ADDRESS_WIDTH_RAM-1 downto 3)
    );
  address_ram_limited(2 downto 0) <= address(2 downto 0);
  
  addressLogic : entity work.nacl_memory_address_logic
    port map (
      switch_state => switch_state,
      address_in   => address_ram_limited(ADDRESS_WIDTH_RAM-1 downto 3),
      address_out  => address_ram(ADDRESS_WIDTH_RAM-1 downto 3)
    );
  address_ram(2 downto 0) <= address_ram_limited(2 downto 0);

  switchBuffer : entity work.nacl_secure_bit_buffer
    port map (
      clk  => clk,
      rst  => rst,
      din  => switch_next,
      dout => switch_state,
      en   => switch_en
    );
  switch_next <= switch_state xor switch_in;
 
  ram : entity work.nacl_memory_ram
    generic map (
      WIDTH   => WORD_SIZE,
      SIZE    => RAM_SIZE,
      SIZE_LD => ADDRESS_WIDTH_RAM
    )
    port map (
      clk  => clk,
      we   => ram_we,
      addr => address_ram,
      di   => data_in,
      do   => data_ram
    );

  ------ ROM --------------------------------------------------------

  romOffsetAdder : entity work.nacl_adder
    generic map(
       WIDTH => ADDRESS_WIDTH_ROM-3
     )
    port map(
      a    => address(ADDRESS_WIDTH_ROM-1 downto 3),
      b    => ADDRESS_ROM_OFFSET(ADDRESS_WIDTH_ROM-4 downto 0),
      sum  => address_rom_int(ADDRESS_WIDTH_ROM-1 downto 3),
      cin  => '1',
      cout => open
     );    
  address_rom_int(2 downto 0) <= address(2 downto 0);

  limiterROM : entity work.nacl_address_limitation
    generic map(
      LIMIT         => NUM_CONSTANTS,
      ADDRESS_WIDTH => ADDRESS_WIDTH_ROM-3
    )
    port map(
      address_in  => address_rom_int(ADDRESS_WIDTH_ROM-1 downto 3),
      address_out => address_rom_limited(ADDRESS_WIDTH_ROM-1 downto 3)
    );
  address_rom_limited(2 downto 0) <= address_rom_int(2 downto 0);

  romAddressMux : entity work.nacl_mux2
    generic map( 
      WIDTH => ADDRESS_WIDTH_ROM
    )
    port map(
      in0  => address_rom_buffer,
      in1  => address_rom_limited,
      outp => address_rom,
      sel  => en_bufALU
    );    

  romAddressBuffer : entity work.nacl_register
    generic map( 
      WIDTH => ADDRESS_WIDTH_ROM
    )
    port map(
      clk  => clk,
      en   => address_rom_buffer_en,
      rst  => rst,
      din  => address_rom_int,
      dout => address_rom_buffer
    );

  rom : entity work.nacl_memory_rom
    port map(
      address_rom => address_rom,
      data_out    => data_rom
    );

  ------ OUTPUT -----------------------------------------------------
  
  outputSelectionLogic : entity work.nacl_memory_output_selection_logic
    port map (
      clk                   => clk,
      rst                   => rst,
      en_bufALU             => en_bufALU,
      address               => address,
      address_rom_buffer_en => address_rom_buffer_en,
      output_sel            => output_sel
    );
    
  outputMux : entity work.nacl_mux2
    generic map( 
      WIDTH => WORD_SIZE
    )
    port map(
      in0  => data_ram,
      in1  => data_rom,
      outp => data_out,
      sel  => output_sel
    );     

end;
