
library ieee;
use ieee.numeric_std.all;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

use work.nacl_constants.all;

entity nacl_selftest is
  port(
    clk             :  in std_logic;
    rst             :  in std_logic;
    start_calc      : out std_logic;
    start_test      :  in std_logic;
    calc_done       :  in std_logic;
    test_done       : out std_logic;
    ram_addr        : out std_logic_vector(ADDRESS_WIDTH-1 downto 0);
    ram_in          : out std_logic_vector(WORD_SIZE-1 downto 0);
    ram_we          : out std_logic;
    ram_data        :  in std_logic_vector(WORD_SIZE-1 downto 0);
    active          : out std_logic;
    error           : out std_logic;
    sel_start_point : out std_logic_vector(START_POINTS_LD-1 downto 0)
  );
end nacl_selftest;

architecture behaviour of nacl_selftest is

  type state_type is (S_WAIT, S_LOAD, S_START, S_CALC, S_FETCH, S_CHECK, S_DONE);
  signal state_current: state_type;
  signal state_next: state_type;

  signal error_int    : std_logic;
  signal reference    : std_logic_vector(WORD_SIZE-1 downto 0);
  signal counter0_out : std_logic_vector(COUNTER_WIDTH-1 downto 0);
  signal counter0_rst : std_logic;
  signal counter0_en  : std_logic;
  signal counter1_out : std_logic_vector(TESTCASES_LD-1 downto 0);
  signal counter1_en  : std_logic;
  signal counter1_max : std_logic;

begin

  counter0 : entity work.nacl_counter
    generic map(
      WIDTH => COUNTER_WIDTH
    )
    port map(
      clk      => clk,
      en       => counter0_en,
      rst      => counter0_rst,
      counter  => counter0_out,
      overflow => open
    );

  counter1 : entity work.nacl_counter
    generic map(
      WIDTH => TESTCASES_LD
    )
    port map(
      clk      => clk,
      en       => counter1_en,
      rst      => rst,
      counter  => counter1_out,
      overflow => open
    );

  counter1Max : process(counter1_out) begin
    if counter1_out = TESTCASES-1 then
      counter1_max <= '1';
    else
      counter1_max <= '0';
    end if;
  end process;

  state : process (clk, rst, state_next) begin
    if rst = '1' then
      state_current <= S_WAIT;
    elsif rising_edge(clk) then
      state_current <= state_next;
    end if;
  end process;

  nsl : process(state_current, start_test, calc_done, counter0_out, counter1_max, error_int) begin
    case state_current is
      when S_LOAD  => if counter0_out = INPUT_VALUES*8-1 then state_next <= S_START;
                      else                                    state_next <= S_LOAD;
                      end if;
      when S_START => state_next <= S_CALC;
      when S_CALC  => if calc_done = '1' then state_next <= S_FETCH;
                      else                    state_next <= S_CALC;
                      end if;
      when S_FETCH => state_next <= S_CHECK;
      when S_CHECK => if counter0_out = OUTPUT_VALUES*8-1 then state_next <= S_DONE;
                      else                                     state_next <= S_CHECK;
                      end if;
      when S_DONE  => if error_int = '1'       then state_next <= S_WAIT;
                      elsif counter1_max = '1' then state_next <= S_WAIT;
                      else                          state_next <= S_LOAD;
                      end if;
        when others  => if start_test = '1' then state_next <= S_LOAD;
                        else                     state_next <= S_WAIT;
                        end if;
    end case;
  end process;

  controller : process(rst, state_current, counter0_out, counter1_max) begin
    if state_current = S_LOAD then
      ram_we <= '1';
    else
      ram_we <= '0';
    end if;

    if state_current = S_START then
      start_calc <= '1';
    else
      start_calc <= '0';
    end if;

    if state_current = S_WAIT then
      test_done <= '1';
    else
      test_done <= '0';
    end if;

    if state_current = S_LOAD then
      counter0_en <= '1';
    elsif state_current = S_FETCH then
      counter0_en <= '1';
    elsif state_current = S_CHECK then
      counter0_en <= '1';
    else
      counter0_en <= '0';
    end if;

    if state_current = S_START or state_current = S_DONE then
      counter0_rst <= '1';
    else
      counter0_rst <= rst;
    end if;

    if state_current = S_DONE and counter1_max = '0' then
      counter1_en <= '1';
    else
      counter1_en <= '0';
    end if;
  end process;

  addressMux : process(state_current, counter0_out, counter1_out) begin
    if COUNTER_WIDTH = 3 then
      if state_current = S_LOAD then
        ram_addr <= input_address_rom(0)  & counter0_out(2 downto 0);
      elsif state_current = S_CHECK then
        ram_addr <= output_address_rom(0) & counter0_out(2 downto 0);
      else
        ram_addr <= (others => '0');
      end if;
    else
      if state_current = S_LOAD then
        ram_addr <= input_address_rom(conv_integer(counter0_out(COUNTER_WIDTH-1 downto 3))+conv_integer(counter1_out)*INPUT_VALUES) & counter0_out(2 downto 0);
      elsif state_current = S_CHECK or state_current = S_FETCH then
        ram_addr <= output_address_rom(conv_integer(counter0_out(COUNTER_WIDTH-1 downto 3))+conv_integer(counter1_out)*OUTPUT_VALUES) & counter0_out(2 downto 0);
      else
        ram_addr <= (others => '0');
      end if;
    end if;
  end process;

  check : process(rst, clk) begin
    if rst = '1' then
      error_int <= '0';
    elsif rising_edge(clk) then
      if state_current = S_CHECK then
        if ram_data /= reference then
          error_int <= '1';
        end if;
      end if;
    end if;
  end process;

  reference <= output_data_rom(conv_integer(counter0_out(OUTPUT_VALUES_LD-1 downto 0)) + conv_integer(counter1_out)*OUTPUT_VALUES*8);
  ram_in <= input_data_rom(conv_integer(counter0_out(INPUT_VALUES_LD-1 downto 0)) + conv_integer(counter1_out)*INPUT_VALUES*8);

  active <= counter0_en;
  error  <= error_int;

  sel_start_point <= test_start_point_rom(conv_integer(counter1_out));

end;
