%{
  open Qhasm
  open Big_int

  let err msg = raise (InvalidAnnotation msg)

  type mix = E of qexp | B of qbexp

  let ase e =
    fun vmap imap fmap pmap ->
      match e vmap imap fmap pmap with
        E e -> e
      | B e -> err("Encountered a Boolean expression " ^ string_of_qbexp e ^ ". An expression is expected.")

  let asb e =
    fun vmap imap fmap pmap ->
      match e vmap imap fmap pmap with
        E e -> err("Encountered an expression " ^ string_of_qexp e ^ ". A Boolean expression is expected.")
      | B e -> e

  let hf fn idx = "$" ^ fn ^ "@" ^ string_of_int idx

  (** Parse an expression with formal variables defined. *)
  let parse_with vmap imap fmap pmap formals e =
    let imap = Hashtbl.copy imap in
    let _ = List.iter (fun formal -> Hashtbl.add imap formal.vname formal.vsize) formals in
    e vmap imap fmap pmap

  (** Parse a Boolean expression with formal variables defined. *)
  let parse_with_b vmap imap fmap pmap formals e =
    let imap = Hashtbl.copy imap in
    let _ = List.iter (fun formal -> Hashtbl.add imap formal.vname formal.vsize) formals in
    e vmap imap fmap pmap

  let rec estimate vmap imap fmap pmap exp =
    match exp with
      QExpConst n -> None
    | QExpCarry -> None
    | QExpVar qvd ->
      begin
        match qvd with
          QVDVar v -> Some v.vsize
        | QVDDeref _ -> Some 64
        | QVDCoef _ -> Some 64
      end
    | QExpNeg e -> estimate vmap imap fmap pmap e
    | QExpNot e -> estimate vmap imap fmap pmap e
    | QExpCast (signed, e, n) -> Some n
    | QExpAdd (e1, e2)
    | QExpSub (e1, e2)
    | QExpMul (e1, e2)
    | QExpAnd (e1, e2)
    | QExpOr (e1, e2)
    | QExpXor (e1, e2)
    | QExpSmod (e1, e2)
    | QExpUmod (e1, e2) ->
      begin
        match estimate vmap imap fmap pmap e1, estimate vmap imap fmap pmap e2 with
          None, None -> None
        | Some n, None
        | None, Some n -> Some n
        | Some n, Some m -> Some (max n m)
      end
    | QExpPow (e, _) -> estimate vmap imap fmap pmap e
    | QExpConcat (e1, e2) ->
      begin
        match estimate vmap imap fmap pmap e1, estimate vmap imap fmap pmap e2 with
          None, None
        | Some _, None
        | None, Some _ -> err "The bit-widths of expressions in concatenation should be specified explicitly."
        | Some n, Some m -> Some (n + m)
      end
    | QExpSll (e, _)
    | QExpSrl (e, _)
    | QExpSra (e, _) -> estimate vmap imap fmap pmap e
    | QExpSlice (_, i, j) -> Some (i - j + 1)
    | QExpApp (fd, _) -> Some fd.svar.vsize
    | QExpIte (_, e1, e2) -> 
      begin
        match estimate vmap imap fmap pmap e1, estimate vmap imap fmap pmap e2 with
          None, None -> None
        | Some n, None
        | None, Some n -> Some n
        | Some n, Some m -> Some (max n m)
      end

  let apply_binary vmap imap fmap pmap f e1 e2 =
    match estimate vmap imap fmap pmap e1, estimate vmap imap fmap pmap e2 with
      None, None
    | None, Some _
    | Some _, None -> f e1 e2
    | Some n, Some m ->
      if n = m then
        f e1 e2
      else
        err (string_of_qexp e1 ^ " and " ^ string_of_qexp e2 ^ " have different bit-widths. " ^
        string_of_qexp e1 ^ ": " ^ string_of_int n ^ "-bits, " ^ string_of_qexp e2 ^ ": " ^ string_of_int m ^ "-bits")
%}

%token <string> NUM
%token <string> VAR
%token <bool * int> CAST
%token LPAREN RPAREN LSQUARE RSQUARE COMMA
%token COLON AUXVAR CONST INV ASSUME ASSERT CUT PREDICATE SMOD UMOD POWER CONJUNCTION DISJUNCTION RARR
%token PLUSEQ MINUSEQ MULTEQ ANDEQ OREQ XOREQ SHIFTLEQ SHIFTREQ
%token PLUS MINUS MULT AND OR XOR QUESTION EQ NE DOT SLT SGT SLE SGE ULT UGT ULE UGE CARRY NOT SHIFTL SHIFTR USHIFTR
%token LOW HIGH
%token UINT64 MEM64 SPLIT TRUE EOF

%right RARR
%left DISJUNCTION
%left CONJUNCTION
%nonassoc EQ NE ULT ULE UGT UGE SLT SLE SGT SGE
%left PLUS MINUS
%left MULT OR XOR AND SMOD UMOD SHIFTL SHIFTR USHIFTR
%left DOT POWER
%right NOT

%start annotations

%type <(string, Qhasm.qtype) Hashtbl.t -> (string, int) Hashtbl.t -> (string, Qhasm.qfun) Hashtbl.t -> (string, Qhasm.qpred) Hashtbl.t -> Qhasm.qannot list> annotations

%%



/* ==================== Annotations ==================== */

annotations:
   annot annotations
   {
     fun vmap imap fmap pmap ->
       let annot = $1 vmap imap fmap pmap in
       let annots = $2 vmap imap fmap pmap in
       annot@annots
   }
 | annot                         { fun vmap imap fmap pmap -> $1 vmap imap fmap pmap }
;

annot:
  CONST consts                { fun vmap imap fmap pmap -> $2 vmap imap fmap pmap }
| AUXVAR auxvars              { fun vmap imap fmap pmap -> $2 vmap imap fmap pmap }
| PREDICATE predicates        { fun vmap imap fmap pmap -> $2 vmap imap fmap pmap }
| INV bexp                    { fun vmap imap fmap pmap -> [ QInvariant ($2 vmap imap fmap pmap) ] }
| ASSUME bexp                 { fun vmap imap fmap pmap -> [ QAssume ($2 vmap imap fmap pmap) ] }
| ASSERT bexp                 { fun vmap imap fmap pmap -> [ QAssert ($2 vmap imap fmap pmap) ] }
| CUT bexp                    { fun vmap imap fmap pmap -> [ QCut ($2 vmap imap fmap pmap) ] }
| CONST error                 { err "Constants are expected after \"const\"." }
| INV error                   { err "A Boolean expression is expected after \"inv\"." }
| PREDICATE error             { err "A Boolean expression is expected after \"predicate\"." }
| ASSUME error                { err "A Boolean expression is expected after \"assume\"." }
| ASSERT error                { err "A Boolean expression is expected after \"assert\"." }
| CUT error                   { err "A Boolean expression is expected after \"cut\"." }
| error                       { err "An annotation must start with const, var, predicate, inv, assume, assert, or cut." }
;



/* ==================== Pre-declared Constants ==================== */

consts:
  exp consts
  {
    fun vmap imap fmap pmap ->
      let c = QConst ($1 vmap imap fmap pmap) in
      let cs = $2 vmap imap fmap pmap in
      c::cs
  }
| exp { fun vmap imap fmap pmap -> [QConst ($1 vmap imap fmap pmap)] }
;



/* ==================== Logical Variables and Functions ==================== */

auxvars:
  auxvar auxvars
  {
    fun vmap imap fmap pmap ->
      let v = $1 vmap imap fmap pmap in
      let vs = $2 vmap imap fmap pmap in
      v::vs
  }
| auxvar { fun vmap imap fmap pmap -> [$1 vmap imap fmap pmap] }
;

auxvar:
  VAR EQ exp
  {
    fun vmap imap fmap pmap ->
      let v = $1 in
      let e = $3 vmap imap fmap pmap in
      if Hashtbl.mem vmap v then
        err ("The auxiliary variable " ^ v ^ " cannot be a program variable.")
      else
        match estimate vmap imap fmap pmap e with
          None -> err ("The bit-width of " ^ string_of_qexp e ^ " cannot be estimated.")
        | Some n ->
          let _ = Hashtbl.add imap v n in
          QAuxVar (mkvar v n, Some e)
  }
| VAR LPAREN formals RPAREN EQ exp
  {
    fun vmap imap fmap pmap ->
      let v = $1 in
      let formals = $3 in
      if Hashtbl.mem vmap v then
        err ("Function " ^ v ^ " is already a program variable.")
      else if Hashtbl.mem imap v then
        err ("Function " ^ v ^ " is already defined as a logical variable or function.")
      else if Hashtbl.mem pmap v then
        err ("Function " ^ v ^ " is already defined as a predicate.")
      else
        let e = parse_with vmap imap fmap pmap formals $6 in
        match estimate vmap imap fmap pmap e with
          None -> err ("The bit-width of " ^ string_of_qexp e ^ " cannot be estimated.")
        | Some n ->
          let _ = Hashtbl.add imap v n in
        (** Remember the bit-widths of the formals. *)
          let _ = List.fold_left (
            fun idx formal -> 
              let _ = Hashtbl.add imap (hf v idx) formal.vsize in
              idx + 1
          ) 0 formals in
          let qfun = {svar = mkvar v n; sformals = formals; sexp = e} in
          let _ = Hashtbl.add fmap v qfun in
          QFunction qfun
  }
| VAR LPAREN formals RPAREN EQ error  { err ("Invalid definition of function " ^ $1 ^ ".") }
| VAR CAST
  {
    fun vmap imap fmap pmap ->
      let v = $1 in
      let n = snd $2 in
      let _ = Hashtbl.add imap v n in
      QAuxVar (mkvar v n, None)
  }
;



/* ==================== Predicates ==================== */

predicates:
  predicate predicates        { fun vmap imap fmap pmap -> ($1 vmap imap fmap pmap)::($2 vmap imap fmap pmap) }
| predicate                   { fun vmap imap fmap pmap -> [$1 vmap imap fmap pmap] }
;

predicate:
  VAR LPAREN formals RPAREN EQ bexp
  {
    fun vmap imap fmap pmap ->
      let v = $1 in
      let formals = $3 in
      if Hashtbl.mem vmap v then
        err ("Predicate " ^ v ^ " is already a program variable.")
      else if Hashtbl.mem imap v then
        err ("Predicate " ^ v ^ " is already defined as a logical variable or function.")
      else if Hashtbl.mem pmap v then
        err ("Predicate " ^ v ^ " is re-defined.")
      else
        let e = parse_with_b vmap imap fmap pmap formals $6 in
          (** Remember the bit-widths of the formals. *)
        let _ = List.fold_left (
          fun idx formal -> 
            let _ = Hashtbl.add imap (hf v idx) formal.vsize in
            idx + 1
        ) 0 formals in
        let qpred = {pvar = mkvar v 1; pformals = formals; pbexp = e} in
        let _ = Hashtbl.add pmap v qpred in
        QPredicate qpred
  }
;



/* ==================== Expressions and Boolean Expressions ==================== */

bexp: mix                     { asb $1 };
exp: mix                      { ase $1 };

mix:
  exp2                        { $1 }
| exp2_b QUESTION exp2_e COLON exp2_e
  {
    fun vmap imap fmap pmap ->
      E (QExpIte ($1 vmap imap fmap pmap, $3 vmap imap fmap pmap, $5 vmap imap fmap pmap))
  }
;

exp2_b: exp2                  { asb $1 };
exp2_e: exp2                  { ase $1 };

exp2:
  atomic                      { $1 }
| exp2 lessop atomics bless %prec ULT
  {
    fun vmap imap fmap pmap ->
      let l = ase $1 vmap imap fmap pmap in
      let uop = $4 vmap imap fmap pmap in
      let helper v = 
        match uop with
          None -> apply_binary vmap imap fmap pmap $2 l v
        | Some cmp -> QBexpAnd ($2 l v, cmp v) in
      match $3 vmap imap fmap pmap with
        [] -> assert false
      | hd::[] -> B (helper hd)
      | hd::tl -> B (List.fold_left (fun res v -> QBexpAnd (res, helper v)) (helper hd) tl)
  }
| exp2 moreop atomics bmore %prec ULT
  {
    fun vmap imap fmap pmap ->
      let l = ase $1 vmap imap fmap pmap in
      let uop = $4 vmap imap fmap pmap in
      let helper v = 
        match uop with
          None -> apply_binary vmap imap fmap pmap $2 l v
        | Some cmp -> QBexpAnd ($2 l v, cmp v) in
      match $3 vmap imap fmap pmap with
        [] -> assert false
      | hd::[] -> B (helper hd)
      | hd::tl -> B (List.fold_left (fun res v -> QBexpAnd (res, helper v)) (helper hd) tl)
  }
| exp2 PLUS exp2
  {
    fun vmap imap fmap pmap -> 
      E (apply_binary vmap imap fmap pmap (fun e1 e2 -> QExpAdd (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 MINUS exp2
  {
    fun vmap imap fmap pmap -> 
      E (apply_binary vmap imap fmap pmap (fun e1 e2 -> QExpSub (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 MULT exp2
  {
    fun vmap imap fmap pmap -> 
      E (apply_binary vmap imap fmap pmap (fun e1 e2 -> QExpMul (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 AND exp2
  {
    fun vmap imap fmap pmap -> 
      E (apply_binary vmap imap fmap pmap (fun e1 e2 -> QExpAnd (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 OR exp2
  {
    fun vmap imap fmap pmap -> 
      E (apply_binary vmap imap fmap pmap (fun e1 e2 -> QExpOr (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 XOR exp2
  {
    fun vmap imap fmap pmap -> 
      E (apply_binary vmap imap fmap pmap (fun e1 e2 -> QExpXor (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 SMOD exp2
  {
    fun vmap imap fmap pmap -> 
      E (apply_binary vmap imap fmap pmap (fun e1 e2 -> QExpSmod (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 UMOD exp2
  {
    fun vmap imap fmap pmap -> 
      E (apply_binary vmap imap fmap pmap (fun e1 e2 -> QExpUmod (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 SHIFTL exp2            { fun vmap imap fmap pmap -> E (QExpSll (ase $1 vmap imap fmap pmap, ase $3 vmap imap fmap pmap)) }
| exp2 USHIFTR exp2           { fun vmap imap fmap pmap -> E (QExpSrl (ase $1 vmap imap fmap pmap, ase $3 vmap imap fmap pmap)) }
| exp2 SHIFTR exp2            { fun vmap imap fmap pmap -> E (QExpSra (ase $1 vmap imap fmap pmap, ase $3 vmap imap fmap pmap)) }
| exp2 POWER exp2             { fun vmap imap fmap pmap -> E (QExpPow (ase $1 vmap imap fmap pmap, ase $3 vmap imap fmap pmap)) }
| MINUS exp2 %prec MINUS
  {
    fun vmap imap fmap pmap ->
      let e = ase $2 vmap imap fmap pmap in
      E (match e with
        QExpConst n -> QExpConst (minus_big_int n)
      | _ -> QExpNeg e)
  }
| exp2 EQ exp2
  { 
    fun vmap imap fmap pmap -> 
      B (apply_binary vmap imap fmap pmap (fun e1 e2 -> QBexpEq (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 NE exp2
  {
    fun vmap imap fmap pmap ->
      B (apply_binary vmap imap fmap pmap (fun e1 e2 -> QBexpNe (e1, e2)) (ase $1 vmap imap fmap pmap) (ase $3 vmap imap fmap pmap))
  }
| exp2 RARR exp2              { fun vmap imap fmap pmap -> B (QBexpImp (asb $1 vmap imap fmap pmap, asb $3 vmap imap fmap pmap)) }
| exp2 DISJUNCTION exp2       { fun vmap imap fmap pmap -> B (QBexpOr (asb $1 vmap imap fmap pmap, asb $3 vmap imap fmap pmap)) }
| exp2 CONJUNCTION exp2       { fun vmap imap fmap pmap -> B (QBexpAnd (asb $1 vmap imap fmap pmap, asb $3 vmap imap fmap pmap)) }
| NOT exp2
  {
    fun vmap imap fmap pmap -> 
      match $2 vmap imap fmap pmap with
        E e -> E (QExpNot e)
      | B e -> B (QBexpNeg e)
  }
;

bless:
  lessop exp2 %prec ULT       { fun vmap imap fmap pmap -> Some (fun v -> $1 v (ase $2 vmap imap fmap pmap)) }
|                             { fun vmap imap fmap pmap -> None }
;

bmore:
  moreop exp2 %prec ULT       { fun vmap imap fmap pmap -> Some (fun v -> $1 v (ase $2 vmap imap fmap pmap)) }
|                             { fun vmap imap fmap pmap -> None }
;

atomics:
  atomic_e                    { fun vmap imap fmap pmap -> [$1 vmap imap fmap pmap] }
| atomic_e COMMA atomics      { fun vmap imap fmap pmap -> ($1 vmap imap fmap pmap)::($3 vmap imap fmap pmap) }
;

atomic_e: atomic              { ase $1 };

atomic:
  big                         { fun vmap imap fmap pmap -> E (QExpConst $1) }
| CARRY                       { fun vmap imap fmap pmap -> E (QExpCarry) }
| var                         { fun vmap imap fmap pmap -> E ($1 vmap imap fmap pmap) }
| LPAREN mix RPAREN           { $2 }
| atomic_e DOT atomic_e       { fun vmap imap fmap pmap -> E (QExpConcat ($1 vmap imap fmap pmap, $3 vmap imap fmap pmap)) }
| atomic_e CAST               { fun vmap imap fmap pmap -> E (QExpCast (fst $2, $1 vmap imap fmap pmap, snd $2)) }
| atomic_e LOW
  {
    fun vmap imap fmap pmap ->
      let e = $1 vmap imap fmap pmap in
      match estimate vmap imap fmap pmap e with
        None -> err ("Unknown bit-width of " ^ string_of_qexp e)
      | Some s -> E (QExpSlice (e, s / 2 - 1, 0))
  }
| atomic_e HIGH
  {
    fun vmap imap fmap pmap ->
      let e = $1 vmap imap fmap pmap in
      match estimate vmap imap fmap pmap e with
        None -> err ("Unknown bit-width of " ^ string_of_qexp e)
      | Some s -> E (QExpSlice (e, s - 1, s / 2))
  }
| atomic_e LSQUARE num COMMA num RSQUARE
  {
    if $3 < $5 then
      err ("The starting index should be greater than or equal to the ending index.")
    else
      fun vmap imap fmap pmap -> E (QExpSlice ($1 vmap imap fmap pmap, $3, $5))
  }
| atomic_e LSQUARE num RSQUARE
  {
    fun vmap imap fmap pmap -> 
      if $3 mod 8 <> 0 then
        err ("The offset of memory access " ^ string_of_qexp ($1 vmap imap fmap pmap) ^ "[" ^ string_of_int $3 ^ "] should be a multiple of 8.")
      else
        match $1 vmap imap fmap pmap with
          QExpVar (QVDVar v) -> E (QExpVar (QVDDeref (v, $3)))
        | _ -> err "Deference can be used in variables only."
  }
| VAR LPAREN actuals RPAREN
  {
    let get_actuals vmap imap fmap pmap fn= 
      let actuals = $3 vmap imap fmap pmap in
      (* Check if the bit-widths of the actuals match the formals. *)
      let _ = List.fold_left (
        fun idx actual ->
          let sf = Hashtbl.find imap (hf fn idx) in
          let sa = size_of_exp actual in
          if sf <> sa then
            err ("The bit-width of the actual " ^ 
                    string_of_qexp actual ^ " is " ^ string_of_int sa ^ 
                    " but the bit-width of the corresponding formal is " ^ string_of_int sf ^ ".")
          else
            idx + 1
      ) 0 actuals in
      actuals in
    fun vmap imap fmap pmap ->
      if Hashtbl.mem fmap $1 then
        let fd = Hashtbl.find fmap $1 in
        let actuals = get_actuals vmap imap fmap pmap fd.svar.vname in
        E (QExpApp (fd, actuals))
      else if Hashtbl.mem pmap $1 then
        let p = Hashtbl.find pmap $1 in
        let actuals = get_actuals vmap imap fmap pmap p.pvar.vname in
        B (QBexpApp (p, actuals))
      else
        err ($1 ^ " is neither a function nor a predicate.")
  }
| TRUE                        { fun vmap imap fmap pmap -> B (QBexpTrue) }
| LPAREN mix RPAREN SPLIT num
  {
    fun vmap imap fmap pmap ->
      let n = $5 in
      match asb $2 vmap imap fmap pmap with
        QBexpEq (e1, e2) ->
          begin
            match estimate vmap imap fmap pmap e1 with
              None -> err "Unable to estimate the bit-width for bit splitting."
            | Some s ->
              if s mod n <> 0 then
                err "The number of splits should be a factor of the bit-width."
              else
                let inc = s / n in
                let rec helper upper lower =
                  if lower < 0 then
                    assert false
                  else if lower == 0 then
                    QBexpEq (QExpSlice (e1, upper, lower), QExpSlice (e2, upper, lower))
                  else
                    QBexpAnd (QBexpEq (QExpSlice (e1, upper, lower), QExpSlice (e2, upper, lower)), helper (upper - inc) (lower - inc)) in
                B (helper (s - 1) (s - inc))
          end
      | _ -> err "Bit splitting is only supported by equality."
  }
| atomic_e LSQUARE error
  {
    fun vmap imap fmap pmap ->
      let e = string_of_qexp ($1 vmap imap fmap pmap) in
      err ("Use \"" ^ e ^ "[n:m]\" to extract a bit-vector from " ^ e ^ ". " ^
              "Use \"" ^ e ^ "[n]\" as a shorthand of mem64[" ^ e ^ " + m] where m = n * 8.")
  }
;


num:
  NUM { int_of_string $1 }
;

big:
  NUM { big_int_of_string $1 }
;

var:
| VAR
    {
      fun vmap imap fmap pmap ->
        if $1 = "carry" then
          QExpCarry
        else
          try
            QExpVar (QVDVar (mkvar $1 (size_of_qtype (Hashtbl.find vmap $1))))
          with Not_found ->
            begin
              try
                QExpVar (QVDVar (mkvar $1 (Hashtbl.find imap $1)))
              with Not_found ->
                err ("Variable " ^ $1 ^ " is not defined.")
            end
    }
/* The following two rules will produce shift/reduce conflicts. */
/*
| MULT LPAREN UINT64 MULT RPAREN LPAREN qvar PLUS num RPAREN { fun vmap imap fmap pmap -> QExpVar (QVDDeref ($7 vmap imap fmap pmap, $9)) }
| MULT LPAREN UINT64 MULT RPAREN AND VAR                     { fun vmap imap fmap pmap -> QExpVar (QVDCoef (mkvar $7 64)) } 
*/
| MEM64 LSQUARE qvar PLUS num RSQUARE                        { fun vmap imap fmap pmap -> QExpVar (QVDDeref ($3 vmap imap fmap pmap, $5)) }
| MEM64 LSQUARE VAR RSQUARE                                  { fun vmap imap fmap pmap -> QExpVar (QVDCoef (mkvar $3 64)) }
| MEM64 error
    {
      err "Use mem64[v + n] to load a value from memory or use mem64[v] to refer to a predefined constant."
    }
;

qvar:
  VAR
  {
    fun vmap imap fmap pmap ->
      try
        mkvar $1 (size_of_qtype (Hashtbl.find vmap $1))
      with Not_found ->
        err ("Variable " ^ $1 ^ " is not defined.")
  }
;



/* ==================== Formal and Actual Parameters ==================== */

formals:
  fparams                     { $1 }
|                             { [] }
;

fparams:
  fparam COMMA fparams        { $1::$3 }
| fparam                      { [ $1 ] }
;

fparam: VAR CAST              { mkvar $1 (snd $2) };

actuals:
  aparams                     { fun vmap imap fmap pmap -> $1 vmap imap fmap pmap }
|                             { fun vmap imap fmap pmap -> [] }
;

aparams:
  exp COMMA aparams           { fun vmap imap fmap pmap -> ($1 vmap imap fmap pmap)::($3 vmap imap fmap pmap)}
| exp                         { fun vmap imap fmap pmap -> [$1 vmap imap fmap pmap] }
;



/* ==================== Comparison Operators ==================== */

lessop:
  ULT                      { fun e1 e2 -> QBexpUlt (e1, e2) }
| ULE                      { fun e1 e2 -> QBexpUle (e1, e2) }
| SLT                      { fun e1 e2 -> QBexpSlt (e1, e2) }
| SLE                      { fun e1 e2 -> QBexpSle (e1, e2) }
;

moreop:
  UGT                      { fun e1 e2 -> QBexpUgt (e1, e2) }
| UGE                      { fun e1 e2 -> QBexpUge (e1, e2) }
| SGT                      { fun e1 e2 -> QBexpSgt (e1, e2) }
| SGE                      { fun e1 e2 -> QBexpSge (e1, e2) }
;
