
(** the type of bit size *)
type bbits = int

(** the type of variables *)
type bvar = int

(** the type of statements *)
type bstmt =
| Bcomment of string
| Bempty
| Bone of bvar * bbits
| Bones of bvar * bbits
| Bzero of bvar * bbits
| Bconstd of bvar * bbits * string
| Bconsth of bvar * bbits * string
| Bconst of bvar * bbits * string
| Bvar of bvar * bbits * string option
| Bnot of bvar * bbits * bvar
| Bneg of bvar * bbits * bvar
| Bredand of bvar * bbits * bvar
| Bredor of bvar * bbits * bvar
| Bredxor of bvar * bbits * bvar
| Binc of bvar * bbits * bvar
| Bdec of bvar * bbits * bvar
| Band of bvar * bbits * bvar * bvar
| Bor of bvar * bbits * bvar * bvar
| Bxor of bvar * bbits * bvar * bvar
| Bnand of bvar * bbits * bvar * bvar
| Bnor of bvar * bbits * bvar * bvar
| Bxnor of bvar * bbits * bvar * bvar
| Bimplies of bvar * bbits * bvar * bvar
| Biff of bvar * bbits * bvar * bvar
| Badd of bvar * bbits * bvar * bvar
| Bsub of bvar * bbits * bvar * bvar
| Bmul of bvar * bbits * bvar * bvar
| Burem of bvar * bbits * bvar * bvar
| Bsrem of bvar * bbits * bvar * bvar
| Budiv of bvar * bbits * bvar * bvar
| Bsdiv of bvar * bbits * bvar * bvar
| Bsmod of bvar * bbits * bvar * bvar
| Beq of bvar * bbits * bvar * bvar
| Bne of bvar * bbits * bvar * bvar
| Bult of bvar * bbits * bvar * bvar
| Bslt of bvar * bbits * bvar * bvar
| Bulte of bvar * bbits * bvar * bvar
| Bslte of bvar * bbits * bvar * bvar
| Bugt of bvar * bbits * bvar * bvar
| Bsgt of bvar * bbits * bvar * bvar
| Bugte of bvar * bbits * bvar * bvar
| Bsgte of bvar * bbits * bvar * bvar
| Bsll of bvar * bbits * bvar * bvar
| Bsrl of bvar * bbits * bvar * bvar
| Bsra of bvar * bbits * bvar * bvar
| Bror of bvar * bbits * bvar * bvar
| Brol of bvar * bbits * bvar * bvar
| Buaddo of bvar * bbits * bvar * bvar
| Bsaddo of bvar * bbits * bvar * bvar
| Busubo of bvar * bbits * bvar * bvar
| Bssubo of bvar * bbits * bvar * bvar
| Bumulo of bvar * bbits * bvar * bvar
| Bsmulo of bvar * bbits * bvar * bvar
| Bsdivo of bvar * bbits * bvar * bvar
| Bconcat of bvar * bbits * bvar * bvar
| Bcond of bvar * bbits * bvar * bvar * bvar
| Bslice of bvar * bbits * bvar * bvar * bvar
| Barray of bvar * bbits * bbits (* element bit-width, array size bit-width *)
| Bacond of bvar * bbits * bbits * bvar * bvar * bvar (* element bit-width, array size bit-width, condition, true-var, false-var *)
| Bwrite of bvar * bbits * bbits * bvar * bvar * bvar (* element bit-width, array size bit-width, array-var, index-var, value-var *)
| Bread of bvar * bbits * bvar * bvar (* bit-width, array-var, index-var *)
| Broot of bvar * bbits * bvar

(** the type of programs *)
type bprog = {
  (** the maximum ID of variables in this program *)
  mutable pmax: int;

  (** a hash table from a variable ID to the declaration of the variable *)
  mutable phash: (int, bstmt) Hashtbl.t; 

  (** the program lines *)
  mutable pstmts: bstmt list
}



let string_of_bbits bits = string_of_int bits

let string_of_bvar var = string_of_int var

let string_of_bstmt stmt =
  match stmt with
    Bcomment c -> "; " ^ c
  | Bempty -> ""
  | Bone (v, bits) -> (string_of_bvar v) ^ " one " ^ (string_of_bbits bits)
  | Bones (v, bits) -> (string_of_bvar v) ^ " ones " ^ (string_of_bbits bits)
  | Bzero (v, bits) -> (string_of_bvar v) ^ " zero " ^ (string_of_bbits bits)
  | Bconstd (v, bits, const) -> (string_of_bvar v) ^ " constd " ^ (string_of_bbits bits) ^ " " ^ const
  | Bconsth (v, bits, const) -> (string_of_bvar v) ^ " consth " ^ (string_of_bbits bits) ^ " " ^ const
  | Bconst (v, bits, const) -> (string_of_bvar v) ^ " const " ^ (string_of_bbits bits) ^ " " ^ const
  | Bvar (v, bits, name) -> 
    let n = 
      match name with 
        None -> ""
      | Some m -> " " ^ m in
    (string_of_bvar v) ^ " var " ^ (string_of_bbits bits) ^ n
  | Bnot (v, bits, var) -> (string_of_bvar v) ^ " not " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar var)
  | Bneg (v, bits, var) -> (string_of_bvar v) ^ " neg " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar var)
  | Bredand (v, bits, var) -> (string_of_bvar v) ^ " redand " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar var)
  | Bredor (v, bits, var) -> (string_of_bvar v) ^ " redor " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar var)
  | Bredxor (v, bits, var) -> (string_of_bvar v) ^ " redxor " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar var)
  | Binc (v, bits, var) -> (string_of_bvar v) ^ " inc " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar var)
  | Bdec (v, bits, var) -> (string_of_bvar v) ^ " dec " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar var)
  | Band (v, bits, v1, v2) -> (string_of_bvar v) ^ " and " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bor (v, bits, v1, v2) -> (string_of_bvar v) ^ " or " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bxor (v, bits, v1, v2) -> (string_of_bvar v) ^ " xor " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bnand (v, bits, v1, v2) -> (string_of_bvar v) ^ " nand " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bnor (v, bits, v1, v2) -> (string_of_bvar v) ^ " nor " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bxnor (v, bits, v1, v2) -> (string_of_bvar v) ^ " xnor " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bimplies (v, bits, v1, v2) -> (string_of_bvar v) ^ " implies " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Biff (v, bits, v1, v2) -> (string_of_bvar v) ^ " iff " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Badd (v, bits, v1, v2) -> (string_of_bvar v) ^ " add " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsub (v, bits, v1, v2) -> (string_of_bvar v) ^ " sub " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bmul (v, bits, v1, v2) -> (string_of_bvar v) ^ " mul " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Burem (v, bits, v1, v2) -> (string_of_bvar v) ^ " urem " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsrem (v, bits, v1, v2) -> (string_of_bvar v) ^ " srem " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Budiv (v, bits, v1, v2) -> (string_of_bvar v) ^ " udiv " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsdiv (v, bits, v1, v2) -> (string_of_bvar v) ^ " sdiv " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsmod (v, bits, v1, v2) -> (string_of_bvar v) ^ " smod " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Beq (v, bits, v1, v2) -> (string_of_bvar v) ^ " eq " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bne (v, bits, v1, v2) -> (string_of_bvar v) ^ " ne " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bult (v, bits, v1, v2) -> (string_of_bvar v) ^ " ult " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bslt (v, bits, v1, v2) -> (string_of_bvar v) ^ " slt " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bulte (v, bits, v1, v2) -> (string_of_bvar v) ^ " ulte " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bslte (v, bits, v1, v2) -> (string_of_bvar v) ^ " slte " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bugt (v, bits, v1, v2) -> (string_of_bvar v) ^ " ugt " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsgt (v, bits, v1, v2) -> (string_of_bvar v) ^ " sgt " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bugte (v, bits, v1, v2) -> (string_of_bvar v) ^ " ugte " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsgte (v, bits, v1, v2) -> (string_of_bvar v) ^ " sgte " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsll (v, bits, v1, v2) -> (string_of_bvar v) ^ " sll " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsrl (v, bits, v1, v2) -> (string_of_bvar v) ^ " srl " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsra (v, bits, v1, v2) -> (string_of_bvar v) ^ " sra " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bror (v, bits, v1, v2) -> (string_of_bvar v) ^ " ror " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Brol (v, bits, v1, v2) -> (string_of_bvar v) ^ " rol " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Buaddo (v, bits, v1, v2) -> (string_of_bvar v) ^ " uaddo " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsaddo (v, bits, v1, v2) -> (string_of_bvar v) ^ " saddo " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Busubo (v, bits, v1, v2) -> (string_of_bvar v) ^ " usubo " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bssubo (v, bits, v1, v2) -> (string_of_bvar v) ^ " ssubo " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bumulo (v, bits, v1, v2) -> (string_of_bvar v) ^ " umulo " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsmulo (v, bits, v1, v2) -> (string_of_bvar v) ^ " smulo " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bsdivo (v, bits, v1, v2) -> (string_of_bvar v) ^ " sdivo " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bconcat (v, bits, v1, v2) -> (string_of_bvar v) ^ " concat " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bcond (v, bits, cond, v1, v2) -> (string_of_bvar v) ^ " cond " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar cond) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bslice (v, bits, var, upper, lower) -> (string_of_bvar v) ^ " slice " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar var) ^ " " ^ (string_of_bvar upper) ^ " " ^ (string_of_bvar lower)
  | Barray (v, bits, size) -> (string_of_bvar v) ^ " array " ^ (string_of_bbits bits) ^ " " ^ (string_of_bbits size)
  | Bacond (v, bits, size, array, v1, v2) -> (string_of_bvar v) ^ " cond " ^ (string_of_bbits bits) ^ " " ^ (string_of_bbits size) ^ " " ^ (string_of_bvar array) ^ " " ^ (string_of_bvar v1) ^ " " ^ (string_of_bvar v2)
  | Bwrite (v, bits, size, array, index, value) -> (string_of_bvar v) ^ " write " ^ (string_of_bbits bits) ^ " " ^ (string_of_bbits size) ^ " " ^ (string_of_bvar array) ^ " " ^ (string_of_bvar index) ^ " " ^ (string_of_bvar value)
  | Bread (v, bits, array, index) -> (string_of_bvar v) ^ " read " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar array) ^ " " ^ (string_of_bvar index)
  | Broot (v, bits, var) -> (string_of_bvar v) ^ " root " ^ (string_of_bbits bits) ^ " " ^ (string_of_bvar var)

let string_of_bprog prog = (String.concat "\n" (List.map string_of_bstmt prog.pstmts)) ^ "\n"

let mkcomment comment = Bcomment comment
let mkempty () = Bempty
let mkone v bits = Bone (v, bits)
let mkones v bits = Bones (v, bits)
let mkzero v bits = Bzero (v, bits)
let mkconstd v bits value = Bconstd (v, bits, value)
let mkconsth v bits value = Bconsth (v, bits, value)
let mkconst v bits value = Bconst (v, bits, value)
let mkvar v bits = Bvar (v, bits, None)
let mkvar2 v bits name = Bvar (v, bits, Some name)
let mknot v bits var = Bnot (v, bits, var)
let mkneg v bits var = Bneg (v, bits, var)
let mkredand v bits var = Bredand (v, bits, var)
let mkredor v bits var = Bredor (v, bits, var)
let mkredxor v bits var = Bredxor (v, bits, var)
let mkinc v bits var = Binc (v, bits, var)
let mkdec v bits var = Bdec (v, bits, var)
let mkand v bits v1 v2 = Band (v, bits, v1, v2)
let mkor v bits v1 v2 = Bor (v, bits, v1, v2)
let mkxor v bits v1 v2 = Bxor (v, bits, v1, v2)
let mknand v bits v1 v2 = Bnand (v, bits, v1, v2)
let mknor v bits v1 v2 = Bnor (v, bits, v1, v2)
let mkxnor v bits v1 v2 = Bxnor (v, bits, v1, v2)
let mkimplies v bits v1 v2 = Bimplies (v, bits, v1, v2)
let mkiff v bits v1 v2 = Biff (v, bits, v1, v2)
let mkadd v bits v1 v2 = Badd (v, bits, v1, v2)
let mksub v bits v1 v2 = Bsub (v, bits, v1, v2)
let mkmul v bits v1 v2 = Bmul (v, bits, v1, v2)
let mkurem v bits v1 v2 = Burem (v, bits, v1, v2)
let mksrem v bits v1 v2 = Bsrem (v, bits, v1, v2)
let mkudiv v bits v1 v2 = Budiv (v, bits, v1, v2)
let mksdiv v bits v1 v2 = Bsdiv (v, bits, v1, v2)
let mksmod v bits v1 v2 = Bsmod (v, bits, v1, v2)
let mkeq v bits v1 v2 = Beq (v, bits, v1, v2)
let mkne v bits v1 v2 = Bne (v, bits, v1, v2)
let mkult v bits v1 v2 = Bult (v, bits, v1, v2)
let mkslt v bits v1 v2 = Bslt (v, bits, v1, v2)
let mkulte v bits v1 v2 = Bulte (v, bits, v1, v2)
let mkslte v bits v1 v2 = Bslte (v, bits, v1, v2)
let mkugt v bits v1 v2 = Bugt (v, bits, v1, v2)
let mksgt v bits v1 v2 = Bsgt (v, bits, v1, v2)
let mkugte v bits v1 v2 = Bugte (v, bits, v1, v2)
let mksgte v bits v1 v2 = Bsgte (v, bits, v1, v2)
let mksll v bits v1 v2 = Bsll (v, bits, v1, v2)
let mksrl v bits v1 v2 = Bsrl (v, bits, v1, v2)
let mksra v bits v1 v2 = Bsra (v, bits, v1, v2)
let mkror v bits v1 v2 = Bror (v, bits, v1, v2)
let mkrol v bits v1 v2 = Brol (v, bits, v1, v2)
let mkuaddo v bits v1 v2 = Buaddo (v, bits, v1, v2)
let mksaddo v bits v1 v2 = Bsaddo (v, bits, v1, v2)
let mkusubo v bits v1 v2 = Busubo (v, bits, v1, v2)
let mkssubo v bits v1 v2 = Bssubo (v, bits, v1, v2)
let mkumulo v bits v1 v2 = Bumulo (v, bits, v1, v2)
let mksmulo v bits v1 v2 = Bsmulo (v, bits, v1, v2)
let mksdivo v bits v1 v2 = Bsdivo (v, bits, v1, v2)
let mkconcat v bits v1 v2 = Bconcat (v, bits, v1, v2)
let mkcond v bits cond v1 v2 = Bcond (v, bits, cond, v1, v2)
let mkslice v bits var upper lower = Bslice (v, bits, var, upper, lower)
let mkarray v bits size = Barray (v, bits, size)
let mkacond v bits size cond v1 v2 = Bacond (v, bits, size, cond, v1, v2)
let mkwrite v bits size array index value = Bwrite (v, bits, size, array, index, value)
let mkread v bits array index = Bread (v, bits, array, index)
let mkroot v bits var = Broot (v, bits, var)

(** 
    * Returns the variable holding the returned value of a statement.
    * If the statement is a comment or an empty line, -1 will be returned.
*)
let stmtvar stmt =
  match stmt with
  | Bcomment _
  | Bempty -> -1
  | Bone (v, _)
  | Bones (v, _)
  | Bzero (v, _)
  | Bconstd (v, _, _)
  | Bconsth (v, _, _)
  | Bconst (v, _, _)
  | Bvar (v, _, _)
  | Bnot (v, _, _)
  | Bneg (v, _, _)
  | Bredand (v, _, _)
  | Bredor (v, _, _)
  | Bredxor (v, _, _)
  | Binc (v, _, _)
  | Bdec (v, _, _)
  | Band (v, _, _, _)
  | Bor (v, _, _, _)
  | Bxor (v, _, _, _)
  | Bnand (v, _, _, _)
  | Bnor (v, _, _, _)
  | Bxnor (v, _, _, _)
  | Bimplies (v, _, _, _)
  | Biff (v, _, _, _)
  | Badd (v, _, _, _)
  | Bsub (v, _, _, _)
  | Bmul (v, _, _, _)
  | Burem (v, _, _, _)
  | Bsrem (v, _, _, _)
  | Budiv (v, _, _, _)
  | Bsdiv (v, _, _, _)
  | Bsmod (v, _, _, _)
  | Beq (v, _, _, _)
  | Bne (v, _, _, _)
  | Bult (v, _, _, _)
  | Bslt (v, _, _, _)
  | Bulte (v, _, _, _)
  | Bslte (v, _, _, _)
  | Bugt (v, _, _, _)
  | Bsgt (v, _, _, _)
  | Bugte (v, _, _, _)
  | Bsgte (v, _, _, _)
  | Bsll (v, _, _, _)
  | Bsrl (v, _, _, _)
  | Bsra (v, _, _, _)
  | Bror (v, _, _, _)
  | Brol (v, _, _, _)
  | Buaddo (v, _, _, _)
  | Bsaddo (v, _, _, _)
  | Busubo (v, _, _, _)
  | Bssubo (v, _, _, _)
  | Bumulo (v, _, _, _)
  | Bsmulo (v, _, _, _)
  | Bsdivo (v, _, _, _)
  | Bconcat (v, _, _, _)
  | Bcond (v, _, _, _, _)
  | Bslice (v, _, _, _, _)
  | Barray (v, _, _)
  | Bacond (v, _, _, _, _, _)
  | Bwrite (v, _, _, _, _, _)
  | Bread (v, _, _, _)
  | Broot (v, _, _) -> v

(** Returns the number of bits of the returned value of a statement. *)
let stmtsize stmt =
  match stmt with
  | Bone (_, bits)
  | Bones (_, bits)
  | Bzero (_, bits)
  | Bconstd (_, bits, _)
  | Bconsth (_, bits, _)
  | Bconst (_, bits, _)
  | Bvar (_, bits, _)
  | Bnot (_, bits, _)
  | Bneg (_, bits, _)
  | Bredand (_, bits, _)
  | Bredor (_, bits, _)
  | Bredxor (_, bits, _)
  | Binc (_, bits, _)
  | Bdec (_, bits, _)
  | Band (_, bits, _, _)
  | Bor (_, bits, _, _)
  | Bxor (_, bits, _, _)
  | Bnand (_, bits, _, _)
  | Bnor (_, bits, _, _)
  | Bxnor (_, bits, _, _)
  | Bimplies (_, bits, _, _)
  | Biff (_, bits, _, _)
  | Badd (_, bits, _, _)
  | Bsub (_, bits, _, _)
  | Bmul (_, bits, _, _)
  | Burem (_, bits, _, _)
  | Bsrem (_, bits, _, _)
  | Budiv (_, bits, _, _)
  | Bsdiv (_, bits, _, _)
  | Bsmod (_, bits, _, _)
  | Beq (_, bits, _, _)
  | Bne (_, bits, _, _)
  | Bult (_, bits, _, _)
  | Bslt (_, bits, _, _)
  | Bulte (_, bits, _, _)
  | Bslte (_, bits, _, _)
  | Bugt (_, bits, _, _)
  | Bsgt (_, bits, _, _)
  | Bugte (_, bits, _, _)
  | Bsgte (_, bits, _, _)
  | Bsll (_, bits, _, _)
  | Bsrl (_, bits, _, _)
  | Bsra (_, bits, _, _)
  | Bror (_, bits, _, _)
  | Brol (_, bits, _, _)
  | Buaddo (_, bits, _, _)
  | Bsaddo (_, bits, _, _)
  | Busubo (_, bits, _, _)
  | Bssubo (_, bits, _, _)
  | Bumulo (_, bits, _, _)
  | Bsmulo (_, bits, _, _)
  | Bsdivo (_, bits, _, _)
  | Bconcat (_, bits, _, _)
  | Bcond (_, bits, _, _, _)
  | Bslice (_, bits, _, _, _)
  | Barray (_, bits, _)
  | Bacond (_, bits, _, _, _, _)
  | Bwrite (_, bits, _, _, _, _)
  | Bread (_, bits, _, _)
  | Broot (_, bits, _) -> bits
  | _ -> 0

(** 
 * Returns the size of a variable in a program.
 * Throws Not_found if the variable is not in the program. 
 *)
let varsize prog var =
  let stmt = Hashtbl.find prog.phash var in
  stmtsize stmt

(** Returns an unused variable ID in a program. *)
let newid prog = prog.pmax + 1

let addstmt prog stmt =
  let _ = prog.pstmts <- prog.pstmts@[stmt] in
  let var = stmtvar stmt in
  if var > 0 then
    let _ = prog.pmax <- max prog.pmax var in
    Hashtbl.add prog.phash var stmt

let addstmts prog stmts =
  List.iter (addstmt prog) stmts

let getstmts prog = prog.pstmts

(** Returns a program. *)
let mkprog stmts = 
  let prog = {
    pmax = 0;
    phash = Hashtbl.create 100;
    pstmts = []
  } in
  let _ = addstmts prog stmts in
  prog
