
open Qhasm
open Big_int

exception Undefined of string
exception Redefined of string
exception InvalidBitWidth of string
exception Unsupported of string

let wordsize = 64

let shiftsize = 6

(** Returns the log of n (base 2) as an integer. *)
let logi n = int_of_float (log (float_of_int n) /. log 2.0)

(** Returns the log of n (base 2) as s string. *)
let logs n = string_of_int (logi n)

let rec genones n =
  if n = 0 then
    ""
  else
    "1" ^ genones (n - 1)

class manager (cmap : (string, string) Hashtbl.t) =
object (self)
  (** the ID of the next Btor variable *)
  val mutable var = 0

  (** a map from a Qhasm variable to the corresponding Btor variable *)
  val mutable vmap : (qvar, Btor.bvar) Hashtbl.t = Hashtbl.create 100

  (** a map from a Qhasm variable to its type *)
  val mutable tmap : (qvar, qtype) Hashtbl.t = Hashtbl.create 100

  (** a map from a bit-width and a binary integer in string to the corresponding Btor variable *)
  val mutable cbmap : (int * string, int) Hashtbl.t = Hashtbl.create 100

  (** a map from a bit-width and a decimal integer in string to the corresponding Btor variable *)
  val mutable cdmap : (int * string, int) Hashtbl.t = Hashtbl.create 100

  (** a map from a bit-width and a hexadecimal integer in string to the corresponding Btor variable *)
  val mutable chmap : (int * string, int) Hashtbl.t = Hashtbl.create 100

  (** a memory *)
  val mutable memory = 0

  (** an assumption *)
  val mutable assumption = 0

  (** an assertion *)
  val mutable assertion = 0

  (** a list of created Btor statements *)
  val mutable stmts = []

  method const v =
    try
      Hashtbl.find cmap v
    with Not_found ->
      raise (Undefined ("The constant " ^ v ^ " is undefined."))

  (** Returns a new ID. *)
  method newvar = 
    let _ = var <- var + 1 in
    var

  (** Sets the type and the bit-width of a Qhasm variable. *)
  method settype qv qt = Hashtbl.add tmap qv qt

  (** Sets the corresponding Btor variable of a Qhasm variable. *)
  method setvar qv bv = Hashtbl.add vmap qv bv

  (**
     * Add a mapping from a Qhasm variable to a Btor variable.
     * A Redefined exception will be raised if the Qhasm has been added.
  *)
  method addvar qv qt bv =
    if Hashtbl.mem vmap qv then
      raise (Redefined ("The variable " ^ qv.vname ^ " is already declared."))
    else
      let _ = self#settype qv qt in
      let _ = self#setvar qv bv in
      ()

  (** Returns the Btor variable of a Qhasm variable. *)
  method getvar qv =
    try
      Hashtbl.find vmap qv
    with Not_found ->
      if Hashtbl.mem tmap qv then
        (** This is a program variable. *)
        raise (Undefined ("The variable " ^ qv.vname ^ " is not defined."))
      else
        (** 
            * This is an instrumentation variable. Create a new variable if the
            * instrumentation variable has not been assigned.
        *)
        let bv = self#newvar in
        let _ = self#addstmt (Btor.mkvar2 bv qv.vsize qv.vname) in
        let _ = self#setvar qv bv in
        bv

  (** Returns the type of a Qhasm variable. *)
  method gettype qv =
    try
      Hashtbl.find tmap qv
    with Not_found ->
      raise (Undefined ("The variable " ^ qv.vname ^ " is not defined."))    

  (** Sets the carry. The bit-width of the carry must be 1. *)
  method setcarry bv = self#setvar (mkvar "carry" 1) bv

  (** Returns the carry. The returned value always has one bit. *)
  method getcarry =
    try
      Hashtbl.find vmap (mkvar "carry" 1)
    with Not_found ->
      let bv = self#genconsti 1 0 in
      let _ = self#setcarry bv in
      bv

  (** Returns the memory. *)
  method getmemory =
    if memory = 0 then
      let bv = self#newvar in
      let _ = self#addstmt (Btor.mkarray bv wordsize wordsize) in
      let _ = memory <- bv in
      bv
    else
      memory

  (** Sets the memory. *)
  method setmemory mem = memory <- mem

  (** Inserts an assumption. *)
  method add_assumption v =
    let bv =
      if assumption = 0 then
        v
      else
        self#genand 1 assumption v in
    let _ = assumption <- bv in
    bv

  (** Inserts an assertion. *)
  method add_assertion v =
    let bv =
      if assertion = 0 then
        v
      else
        self#genand 1 assertion v in
    let _ = assertion <- bv in
    bv

  (** Generates the root as assumption /\ ~assertion. *)
  method genroot =
    let _ = self#gencomment "Root" in
    let assumption = if assumption = 0 then self#genconsti 1 1 else assumption in
    let assertion = if assertion = 0 then self#genconsti 1 1 else assertion in
    let imp = self#genand 1 assumption (self#gennot 1 assertion) in
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkroot bv 1 imp) in
    bv

  (** 
      * Casts a Btor variable to a specified bit-width. Both signed and unsigned variables are considered.
      * s1: the original bit-width
      * s2: the desired bit-width
  *)
  method cast bv s1 s2 =
    if s1 < s2 then
      let zero = self#genzero s1 in
      let cond = self#gensgte bv zero in
      self#gencond s2 cond 
        (self#genconcat s2 (self#genzero (s2 - s1)) bv)
        (self#genconcat s2 (self#genconst (s2 - s1) (genones (s2 - s1))) bv)
    else if s1 > s2 then
      self#genslice s2 bv (s2 - 1) 0
    else
      bv

  (** 
      * Casts a Btor variable to a specified bit-width. Only unsigned variables are considered.
      * s1: the original bit-width
      * s2: the desired bit-width
  *)
  method ucast bv s1 s2 =
    if s1 < s2 then
      self#genconcat s2 (self#genzero (s2 - s1)) bv
    else if s1 > s2 then
      self#genslice s2 bv (s2 - 1) 0
    else
      bv

  (** Inserts a Btor statement. *)
  method addstmt stmt = stmts <- stmts@[stmt]

  (** Returns all Btor statements. *)
  method getstmts = stmts

  (** Removes all Btor statements. *)
  method clear = stmts <- []

  (** Generates bv1 + bv2. *)
  method genadd size bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkadd bv size bv1 bv2) in
    bv

  (** Generates bv1 & bv2. *)
  method genand size bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkand bv size bv1 bv2) in
    bv

  method gencomment comment = self#addstmt (Btor.mkcomment comment)

  (** Generates bv1.bv2 (the concatenation of bv1 and bv2). *)
  method genconcat size bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkconcat bv size bv1 bv2) in
    bv

  (** Generates cond ? bv1 : bv2. *)
  method gencond size cond bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkcond bv size cond bv1 bv2) in
    bv

  (** Generates a binary constant. *)
  method genconst size c =
    try
      Hashtbl.find cbmap (size, c)
    with Not_found ->
      let bv = self#newvar in
      let _ = self#addstmt (Btor.mkconst bv size c) in
      let _ = Hashtbl.add cbmap (size, c) bv in
      bv

  (** Generates a constant. *)
  method genconstd size c =
    try
      Hashtbl.find cdmap (size, c)
    with Not_found ->
      if c.[0] = '-' then
        let bv = self#newvar in
        let _ = self#addstmt (Btor.mkconstd bv size (String.sub c 1 ((String.length c) - 1))) in
        let bv = self#genneg size bv in
        let _ = Hashtbl.add cdmap (size, c) bv in
        bv
      else
        let bv = self#newvar in
        let _ = self#addstmt (Btor.mkconstd bv size c) in
        let _ = Hashtbl.add cdmap (size, c) bv in
        bv

  (** Generates a constant. *)
  method genconsth size c =
    try
      Hashtbl.find chmap (size, c)
    with Not_found ->
      let bv = self#newvar in
      let _ = self#addstmt (Btor.mkconsth bv size c) in
      let _ = Hashtbl.add chmap (size, c) bv in
      bv

  (** Generates an integer constant. *)
  method genconsti size c = self#genconstd size (string_of_int c)

  method genempty = self#addstmt (Btor.mkempty ())

  (** Generates bv1 -> bv2. *)
  method genimplies size bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkimplies bv size bv1 bv2) in
    bv

  (** Generates bv1 = bv2. *)
  method geneq bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkeq bv 1 bv1 bv2) in
    bv

  (** Generates x % m. *)
  method gensmod size x m =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mksmod bv size x m) in
    bv

  (** Generates x %u m. The bit-width of x and m must be (size - 1). *)
  method genumod size x m =
    let bv = self#newvar in
    let x = self#ucast x (size - 1) size in
    let m = self#ucast m (size - 1) size in
    let _ = self#addstmt (Btor.mksmod bv size x m) in
    bv

  (** Generates bv1 * bv2. *)
  method genmul size bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkmul bv size bv1 bv2) in
    bv

  (** Generates bv1 != bv2. *)
  method genne bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkne bv 1 bv1 bv2) in
    bv

  (** Generates -v. *)
  method genneg size v =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkneg bv size v) in
    bv

  (** Generates bv1 | bv2. *)
  method genor size bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkor bv size bv1 bv2) in
    bv

  (** Generates ~v. *)
  method gennot size v =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mknot bv size v) in
    bv

  (** Generates a read of an array. *)
  method genread size array offset =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkread bv size array offset) in
    bv

  (** Generates bv1 > bv2. *)
  method gensgt bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mksgt bv 1 bv1 bv2) in
    bv

  (** Generates bv1 >= bv2. *)
  method gensgte bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mksgte bv 1 bv1 bv2) in
    bv

  (** Generates v[u:l] for a Btor variable v. *)
  method genslice size v u l =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkslice bv size v u l) in
    bv

  (** 
      * Generates v << n.
      * The bit-width of n must be the log of the bit-width of v.
  *)
  method gensll size v n =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mksll bv size v n) in
    bv

  (** Generates bv1 < bv2. *)
  method genslt bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkslt bv 1 bv1 bv2) in
    bv

  (** Generates bv1 <= bv2. *)
  method genslte bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkslte bv 1 bv1 bv2) in
    bv

  (** 
      * Generates v >> n.
      * The bit-width of n must be the log of the bit-width of v.
  *)
  method gensra size v n =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mksra bv size v n) in
    bv

  (** 
      * Generates v >> n.
      * The bit-width of n must be the log of the bit-width of v.
  *)
  method gensrl size v n =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mksrl bv size v n) in
    bv

  (** Generates bv1 - bv2. *)
  method gensub size bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mksub bv size bv1 bv2) in
    bv

  (** Generates bv0 + bv1 + ... + bvn for a list [bv0, bv1, ..., bvn] of Btor variables. *)
  method gensum size bvs =
    match bvs with
      [] -> self#genconsti size 0
    | hd::tl -> List.fold_left (fun res bv -> self#genadd size res bv) hd tl

  (** Generates bv1 > bv2. *)
  method genugt bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkugt bv 1 bv1 bv2) in
    bv

  (** Generates bv1 >= bv2. *)
  method genugte bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkugte bv 1 bv1 bv2) in
    bv

  (** Generates bv1 < bv2. *)
  method genult bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkult bv 1 bv1 bv2) in
    bv

  (** Generates bv1 <= bv2. *)
  method genulte bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkulte bv 1 bv1 bv2) in
    bv

  (** Generates a new Btor variable. *)
  method genvar size =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkvar bv size) in
    bv

  (** Generates a new Btor variable. *)
  method genqvar qv qt =
    let bv = self#newvar in
    let size = size_of_qtype qt in
    let _ = self#addstmt (Btor.mkvar2 bv size qv.vname) in
    let _ = self#addvar qv qt bv in
    bv

  (** Generates a write to an array. *)
  method genwrite size bits array offset value =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkwrite bv size bits array offset value) in
    bv

  (** Generates bv1 ^ bv2. *)
  method genxor size bv1 bv2 =
    let bv = self#newvar in
    let _ = self#addstmt (Btor.mkxor bv size bv1 bv2) in
    bv

  (** Generates 0 of a specified bit-width. *)
  method genzero size =
    try
      Hashtbl.find cdmap (size, "0")
    with Not_found ->
      let bv = self#newvar in
      let _ = self#addstmt (Btor.mkzero bv size) in
      let _ = Hashtbl.add cdmap (size, "0") bv in
      bv
end




let genaddr m addr =
  match addr with
    QAddrBO (base, offset) -> m#gensum wordsize [m#getvar base; m#genconsti wordsize offset]
  | QAddrBI (base, index) -> m#gensum wordsize [m#getvar base; m#getvar index]
  | QAddrBIS (base, index) -> m#gensum wordsize [m#getvar base; m#genmul wordsize (m#getvar index) (m#genconsti wordsize 8)]
  | QAddrBOIS (base, offset, index) -> m#gensum wordsize [m#getvar base; m#genconsti wordsize offset; m#genmul wordsize (m#getvar index) (m#genconsti wordsize 8)]

let read m addr =
  m#genread wordsize m#getmemory (genaddr m addr)

let write m addr value =
  m#setmemory (m#genwrite wordsize wordsize m#getmemory (genaddr m addr) value)

let genvar m qt qv =
  let _ = m#genqvar qv qt in
  ()

let genconstvar m cv =
  match cv with
    QIVConst n -> m#genconsti wordsize n
  | QIVVar v -> m#getvar v

let genvarderef m vd =
  match vd with
    QVDVar v -> m#getvar v
  | QVDDeref (base, offset) -> read m (QAddrBO (base, offset))
  | QVDCoef co -> 
    let v = m#const co.vname in
    if String.length v > 2 && String.sub v 0 2 = "0x" then
      m#genconsth wordsize (String.sub v 2 (String.length v - 2))
    else
      m#genconstd wordsize v

let genload m qv qt addr =
  let bv = read m addr in
  match qt with
    QCastInt8 ->
      let bv = m#cast (m#genslice 8 bv 7 0) 8 wordsize in
      m#setvar qv bv
  | QCastUInt8 ->
      let bv = m#ucast (m#genslice 8 bv 7 0) 8 wordsize in
      m#setvar qv bv
  | QCastInt16 ->
      let bv = m#cast (m#genslice 16 bv 15 0) 16 wordsize in
      m#setvar qv bv
  | QCastUInt16 ->
      let bv = m#ucast (m#genslice 16 bv 15 0) 16 wordsize in
      m#setvar qv bv
  | QCastInt32
  | QCastUInt32
  | QCastInt64
  | QCastUInt64 -> m#setvar qv bv

let genstore m qt addr cv =
  let size = size_of_qtypec qt in
  let bv = genconstvar m cv in
  write m addr (m#cast (m#cast bv wordsize size) size wordsize)

let genexpr m expr =
  match expr with
    QExprConst n -> m#genconsti wordsize n
  | QExprVar v -> m#getvar v
  | QExprCarry -> m#ucast m#getcarry 1 wordsize
  | QExprAddVarVar (v1, v2) -> m#gensum wordsize [m#getvar v1; m#getvar v2]
  | QExprAddVarVarConst (v1, v2, n) -> m#gensum wordsize [m#getvar v1; m#getvar v2; m#genconsti wordsize n]
  | QExprAddVarVarVar (v1, v2, v3) -> m#gensum wordsize [m#getvar v1; m#getvar v2; m#getvar v3]
  | QExprAddVarVarCarry (v1, v2) -> m#gensum wordsize [m#getvar v1; m#getvar v2; m#ucast m#getcarry 1 wordsize]
  | QExprMulVarConst (v, n) -> m#genmul wordsize (m#getvar v) (m#genconsti wordsize n)
  | QExprMulVarCarry v -> m#genmul wordsize (m#getvar v) (m#ucast m#getcarry 1 wordsize)

let genassign m qv expr =
  let bv = genexpr m expr in
  m#setvar qv bv

let genassignifcarry m qv expr neg =
  let c = if neg then m#gennot 1 m#getcarry else m#getcarry in
  let bv = m#gencond qv.vsize c (genexpr m expr) (m#getvar qv) in
  m#setvar qv bv

let gencoef m qv co =
  let bv = genvarderef m (QVDCoef co) in
  m#setvar qv bv

let genaddexpr m ?carry:(carry=false) expr =
  let size = if carry then wordsize + 1 else wordsize in
  match expr with
    QAddExprConst n -> m#genconsti size n
  | QAddExprVar v -> m#ucast (m#getvar v) wordsize size
  | QAddExprCarry -> m#ucast m#getcarry 1 size
  | QAddExprDeref (base, offset) -> m#ucast (read m (QAddrBO (base, offset))) wordsize size
  | QAddExprConstCarry n -> m#gensum size [m#genconsti size n; m#ucast m#getcarry 1 size]
  | QAddExprVarConst (v, n) -> m#gensum size [m#ucast (m#getvar v) wordsize size; m#genconsti size n]
  | QAddExprVarCarry v -> m#gensum size [m#ucast (m#getvar v) wordsize size; m#ucast m#getcarry 1 size]
  | QAddExprDerefCarry (base, offset) -> m#gensum size [m#ucast (read m (QAddrBO (base, offset))) wordsize size; m#ucast m#getcarry 1 size]
  | QAddExprCoef v -> m#ucast (genvarderef m (QVDCoef v)) wordsize size

let genadd m qv expr =
  let bv = m#gensum wordsize [m#getvar qv; genaddexpr m expr] in
  m#setvar qv bv

let genaddcarry m qv expr =
  let lv = m#ucast (m#getvar qv) wordsize (wordsize + 1) in
  let rv = genaddexpr m ~carry:true expr in
  let sum = m#genadd (wordsize + 1) lv rv in
  let carry = m#genslice 1 sum wordsize wordsize in
  let bv = m#genslice wordsize sum (wordsize - 1) 0 in
  let _ = m#setcarry carry in
  m#setvar qv bv

let gensubexpr m ?carry:(carry=false) expr =
  let size = if carry then wordsize + 1 else wordsize in
  match expr with
    QSubExprConst n -> m#genconsti size n
  | QSubExprVar v -> m#ucast (m#getvar v) wordsize size
  | QSubExprCarry -> m#ucast m#getcarry 1 size
  | QSubExprDeref (base, offset) -> m#ucast (read m (QAddrBO (base, offset))) wordsize size
  | QSubExprVarCarry v -> m#genadd size (m#ucast (m#getvar v) wordsize size) (m#ucast m#getcarry 1 size)
  | QSubExprDerefCarry (base, offset) -> m#genadd size (m#ucast (read m (QAddrBO (base, offset))) wordsize size) (m#ucast m#getcarry 1 size)

let gensub m qv expr =
  let bv = m#gensub wordsize (m#getvar qv) (gensubexpr m expr) in
  m#setvar qv bv

let gensubcarry m qv expr =
  let lv = m#ucast (m#getvar qv) wordsize (wordsize + 1) in
  let rv = gensubexpr m ~carry:true expr in
  let sum = m#gensub (wordsize + 1) lv rv in
  let carry = m#genslice 1 sum wordsize wordsize in
  let bv = m#genslice wordsize sum (wordsize - 1) 0 in
  let _ = m#setcarry carry in
  m#setvar qv bv

let genmul m qv expr =
  let bv = m#genmul wordsize (m#getvar qv) (genconstvar m expr) in
  m#setvar qv bv

let genand m qv expr =
  let bv = m#genand wordsize (m#getvar qv) (genvarderef m expr) in
  m#setvar qv bv

let genor m qv expr =
  let bv = m#genor wordsize (m#getvar qv) (genvarderef m expr) in
  m#setvar qv bv

let genxor m qv expr =
  let bv = m#genxor wordsize (m#getvar qv) (genvarderef m expr) in
  m#setvar qv bv

let genconcatmul m signed qv1 qv2 expr =
  let cast = if signed then m#cast else m#ucast in
  let mul = m#genmul (wordsize * 2) 
    (cast (m#getvar qv2) wordsize (wordsize * 2))
    (cast (genvarderef m expr) wordsize (wordsize * 2)) in
  let bv1 = m#genslice wordsize mul (wordsize * 2 - 1) wordsize in
  let bv2 = m#genslice wordsize mul (wordsize - 1) 0 in
  let _ = m#setvar qv1 bv1 in
  let _ = m#setvar qv2 bv2 in
  ()  

let genneg m qv = 
  m#setvar qv (m#genneg wordsize (m#getvar qv))

let gennot m qv = 
  m#setvar qv (m#gennot wordsize (m#getvar qv))

let genshiftsize m size expr =
  match expr with
    QIVConst n -> m#genconsti size n
  | QIVVar v -> m#cast (m#getvar v) wordsize size

let genconcatshiftleft m qv1 qv2 expr =
  let bv = m#genslice wordsize 
    (m#gensll (wordsize * 2) 
       (m#genconcat (wordsize * 2) (m#getvar qv1) (m#getvar qv2)) 
       (genshiftsize m (shiftsize + 1) expr))
    (wordsize * 2 - 1)
    wordsize in
  m#setvar qv1 bv

let genshiftleft m qv expr =
  let bv = m#gensll wordsize (m#getvar qv) (genshiftsize m shiftsize expr) in
  m#setvar qv bv

let genconcatshiftright m qv1 qv2 expr =
  let bv = m#genslice wordsize 
    (m#gensrl (wordsize * 2) 
       (m#genconcat (wordsize * 2) (m#getvar qv2) (m#getvar qv1)) 
       (genshiftsize m (shiftsize + 1) expr))
    (wordsize - 1)
    0 in
  m#setvar qv1 bv

let genshiftright m signed qv expr =
  let sr = if signed then m#gensra else m#gensrl in
  let bv = sr wordsize (m#getvar qv) (genshiftsize m shiftsize expr) in
  m#setvar qv bv

let rec genexp m size exp =
  if pure exp then
    m#genconstd size (string_of_big_int (eval exp))
  else
    match exp with
      QExpConst n -> m#genconstd size (string_of_big_int n)
    | QExpCarry -> m#ucast m#getcarry 1 size
    | QExpVar vd -> genvarderef m vd
    | QExpNeg e -> m#genneg size (genexp m size e)
    | QExpNot e -> m#gennot size (genexp m size e)
    | QExpCast (signed, e, s) -> 
      begin
        if pure e then
          m#genconstd (max size s) (string_of_big_int (eval e))
        else
          (if signed then m#cast else m#ucast) (genexp m (size_of_exp e) e) (size_of_exp e) s
      end
    | QExpAdd (e1, e2) -> m#genadd size (genexp m size e1) (genexp m size e2)
    | QExpSub (e1, e2) -> m#gensub size (genexp m size e1) (genexp m size e2)
    | QExpMul (e1, e2) -> m#genmul size (genexp m size e1) (genexp m size e2)
    | QExpAnd (e1, e2) -> m#genand size (genexp m size e1) (genexp m size e2)
    | QExpOr (e1, e2) -> m#genor size (genexp m size e1) (genexp m size e2)
    | QExpXor (e1, e2) -> m#genxor size (genexp m size e1) (genexp m size e2)
    | QExpSmod (e1, e2) -> m#gensmod size (genexp m size e1) (genexp m size e2)
    | QExpUmod (e1, e2) -> m#genumod size (genexp m (size - 1) e1) (genexp m (size - 1) e2)
    | QExpPow (e, n) -> 
      let n = eval_int n in
      let rec helper res base n =
        if n = 1 then
          res
        else
          helper (m#genmul size res base) base (n - 1) in
      if n < 0 then
        assert false
      else if n = 0 then
        m#genconsti size 1
      else
        let base = genexp m size e in
        helper base base n
    | QExpConcat (e1, e2) -> m#genconcat size (genexp m (size_of_exp e1) e1) (genexp m (size_of_exp e2) e2)
    | QExpSll (e1, e2) -> 
      let s2 = size_of_exp e2 in
      m#gensll size (genexp m (size_of_exp e1) e1) (m#cast (genexp m s2 e2) s2 (logi size))
    | QExpSrl (e1, e2) ->
      let s2 = size_of_exp e2 in
      m#gensrl size (genexp m (size_of_exp e1) e1) (m#cast (genexp m s2 e2) s2 (logi size))
    | QExpSra (e1, e2) ->
      let s2 = size_of_exp e2 in
      m#gensra size (genexp m (size_of_exp e1) e1) (m#cast (genexp m s2 e2) s2 (logi size))
    | QExpSlice (e, i, j) -> m#genslice size (genexp m (size_of_exp e) e) i j
    | QExpApp (fd, actuals) -> genexp m size ((mkfunctor fd.sexp fd.sformals) actuals)
    | QExpIte (b, e1, e2) -> m#gencond size (genbexp m b) (genexp m size e1) (genexp m size e2)

and genexps m es = 
  let max = List.fold_left (fun res e -> max res (size_of_exp e)) 0 es in
  (max, List.map (fun e -> genexp m max e) es)

and genbexp (m : manager) exp =
  let helper f e1 e2 =
    let size = max (size_of_exp e1) (size_of_exp e2) in
    let bv1 = genexp m size e1 in
    let bv2 = genexp m size e2 in
    f bv1 bv2 in
  match exp with
    QBexpTrue -> m#genconsti 1 1
  | QBexpEq (e1, e2) -> helper m#geneq e1 e2
  | QBexpNe (e1, e2) -> helper m#genne e1 e2
  | QBexpSlt (e1, e2) -> helper m#genslt e1 e2
  | QBexpSle (e1, e2) -> helper m#genslte e1 e2
  | QBexpSgt (e1, e2) -> helper m#gensgte e1 e2
  | QBexpSge (e1, e2) -> helper m#gensgte e1 e2
  | QBexpUlt (e1, e2) -> helper m#genult e1 e2
  | QBexpUle (e1, e2) -> helper m#genulte e1 e2
  | QBexpUgt (e1, e2) -> helper m#genugt e1 e2
  | QBexpUge (e1, e2) -> helper m#genugte e1 e2
  | QBexpNeg e -> m#gennot 1 (genbexp m e)
  | QBexpAnd (e1, e2) -> m#genand 1 (genbexp m e1) (genbexp m e2)
  | QBexpOr (e1, e2) -> m#genor 1 (genbexp m e1) (genbexp m e2)
  | QBexpImp (e1, e2) -> m#genimplies 1 (genbexp m e1) (genbexp m e2)
  | QBexpApp (p, actuals) -> genbexp m ((mkfunctor_b p.pbexp p.pformals) actuals)

let genannot (m : manager) annot =
  match annot with
    QAuxVar (qv, eop) ->
      begin
        match eop with
          None -> ignore(m#getvar qv)
        | Some e -> ignore(m#setvar qv (genexp m (size_of_exp e) e))
      end
  | QConst e -> ignore(genexp m (size_of_exp e) e)
  | QFunction _ -> ()
  | QPredicate _ -> ()
  | QInvariant e -> assert false
  | QAssume expr -> let _ = m#add_assumption (genbexp m expr) in ()
  | QAssert expr -> let _ = m#add_assertion (genbexp m expr) in ()
  | QCut _ -> print_endline "The cut should be replaced by assume and assert."; assert false

let genstmt m stmt =
  let _ = m#gencomment (string_of_int stmt.sline ^ ": " ^ string_of_qstmt stmt) in
  let _ = 
    match stmt.skind with
      QVar (qt, qv) -> genvar m qt qv
    | QLoad (qv, qt, addr) -> genload m qv qt addr
    | QStore (qt, addr, cv) -> genstore m qt addr cv
    | QAssign (qv, expr) -> genassign m qv expr
    | QAssignIfCarry (qv, expr, neg) -> genassignifcarry m qv expr neg
    | QCoef (qv, co) -> gencoef m qv co
    | QAdd (qv, expr) -> genadd m qv expr
    | QAddCarry (qv, expr) -> genaddcarry m qv expr
    | QSub (qv, expr) -> gensub m qv expr
    | QSubCarry (qv, expr) -> gensubcarry m qv expr
    | QMul (qv, expr) -> genmul m qv expr
    | QAnd (qv, expr) -> genand m qv expr
    | QOr (qv, expr) -> genor m qv expr
    | QXor (qv, expr) -> genxor m qv expr
    | QConcatMul (signed, qv1, qv2, expr) -> genconcatmul m signed qv1 qv2 expr
    | QNeg qv -> genneg m qv
    | QNot qv -> gennot m qv
    | QConcatShiftLeft (qv1, qv2, expr) -> genconcatshiftleft m qv1 qv2 expr
    | QShiftLeft (qv, expr) -> genshiftleft m qv expr
    | QConcatShiftRight (qv1, qv2, expr) -> genconcatshiftright m qv1 qv2 expr
    | QShiftRight (signed, qv, expr) -> genshiftright m signed qv expr
    | QInput _
    | QCaller _
    | QEnter _
    | QLeave
    | QComment _ -> ()
    | QAnnot annot -> genannot m annot in
  match stmt.skind with
    QComment _ -> ()
  | _ -> m#genempty
  
(**
   * Returns a Btor program as the verification condition of an annotated Qhasm program.
   * The first argument is a map from names of predefined constants to their values.
*)
let generate cmap prog = 
  let m = new manager cmap in
  let _ = List.iter (genstmt m) prog in
  let _ = m#genroot in
  let bp = Btor.mkprog (m#getstmts) in
  bp
