(**
   * CVC3 (http://www.cs.nyu.edu/acsys/cvc3/)
*)

let rec zeros n =
  if n = 0 then
    ""
  else
    "0" ^ zeros (n - 1)

(** A variable in CVC3. *)
type var = string

(** Constants. *)
type const =
  Bin of string
| Hex of string

(** An expression in CVC3. *)
type exp =
  V of var
| Const of const
| Concat of exp * exp
| Extract of exp * int * int
| Sll of exp * exp
| Srl of exp * exp
| Sra of exp * exp
| Sextend of exp * int
| Uextend of exp * int
| Not of exp
| And of exp * exp
| Or of exp * exp
| Xor of exp * exp
| Nand of exp * exp
| Nor of exp * exp
| Xnor of exp * exp
| Neg of exp
| Add of int * exp * exp
| Sub of int * exp * exp
| Mul of int * exp * exp
| Div of int * exp * exp
| Sdiv of int * exp * exp
| Mod of int * exp * exp
| Smod of int * exp * exp
| Ifte of bexp * exp * exp
| Read of exp * exp
| Write of exp * exp * exp

(** A Boolean expression in CVC3. *)
and bexp =
  True
| False
| Eq of exp * exp
| Lt of exp * exp
| Le of exp * exp
| Gt of exp * exp
| Ge of exp * exp
| Slt of exp * exp
| Sle of exp * exp
| Sgt of exp * exp
| Sge of exp * exp
| Band of bexp * bexp
| Bor of bexp * bexp
| Bnot of bexp

(** Statements in CVC3. *)
type stmt = 
  Var of var * int
| Array of var * int * int
| Assert of bexp
| Query of bexp
| Comment of string
| Empty

(** A program in CVC3. *)
type prog = stmt list

let rec enclose e =
  match e with
    V _
  | Const _ -> string_of_exp e
  | _ -> "(" ^ string_of_exp e ^ ")"

and benclose e =
  match e with
    Band _
  | Bor _ -> "(" ^ string_of_bexp e ^ ")"
  | _ -> string_of_bexp e

and string_of_exp e =
  match e with
    V v -> v
  | Const c -> 
    begin
      match c with 
        Bin n -> "0bin" ^ n
      | Hex n -> "0hex" ^ n
    end
  | Concat (e1, e2) -> enclose e1 ^ "@" ^ enclose e2 
  | Extract (e, i, j) -> enclose e ^ "[" ^ string_of_int i ^ ":" ^ string_of_int j ^ "]"
  | Sll (e, n) -> "BVSHL(" ^ string_of_exp e ^ ", " ^ string_of_exp n ^ ")"
  | Srl (e, n) -> "BVLSHR(" ^ string_of_exp e ^ ", " ^ string_of_exp n ^ ")"
  | Sra (e, n) -> "BVASHR(" ^ string_of_exp e ^ ", " ^ string_of_exp n ^ ")"
  | Sextend (e, i) -> "SX(" ^ string_of_exp e ^ ", " ^ string_of_int i ^ ")"
  | Uextend (e, i) -> "BVZEROEXTEND(" ^ string_of_exp e ^ ", " ^ string_of_int i ^ ")"
  | Not e -> "~" ^ enclose e
  | And (e1, e2) -> enclose e1 ^ " & " ^ enclose e2
  | Or (e1, e2) -> enclose e1 ^ " | " ^ enclose e2
  | Xor (e1, e2) -> "BVXOR(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Nand (e1, e2) -> "BVNAND(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Nor (e1, e2) -> "BVNOR(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Xnor (e1, e2) -> "BVXNOR(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Neg e -> "BVUMINUS(" ^ string_of_exp e ^ ")"
  | Add (i, e1, e2) -> "BVPLUS(" ^ string_of_int i ^ ", " ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Sub (i, e1, e2) -> "BVSUB(" ^ string_of_int i ^ ", " ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Mul (i, e1, e2) -> "BVMULT(" ^ string_of_int i ^ ", " ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Div (i, e1, e2) -> "BVDIV(" ^ string_of_int i ^ ", " ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Sdiv (i, e1, e2) -> "SBVDIV(" ^ string_of_int i ^ ", " ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Mod (i, e1, e2) -> "BVMOD(" ^ string_of_int i ^ ", " ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Smod (i, e1, e2) -> "SBVMOD(" ^ string_of_int i ^ ", " ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Ifte (b, e1, e2) -> "IF " ^ string_of_bexp b ^ " THEN " ^ string_of_exp e1 ^ " ELSE " ^ string_of_exp e2
  | Read (e, i) -> enclose e ^ "[" ^ string_of_exp i ^ "]"
  | Write (e, i, v) -> enclose e ^ " WITH [" ^ string_of_exp i ^ "] := " ^ string_of_exp v

and string_of_bexp e =
  match e with
    True -> "TRUE"
  | False -> "FALSE"
  | Eq (e1, e2) -> enclose e1 ^ " = " ^ enclose e2
  | Lt (e1, e2) -> "BVLT(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Le (e1, e2) -> "BVLE(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Gt (e1, e2) -> "BVGT(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Ge (e1, e2) -> "BVGE(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Slt (e1, e2) -> "SBVLT(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Sle (e1, e2) -> "SBVLE(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Sgt (e1, e2) -> "SBVGT(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Sge (e1, e2) -> "SBVGE(" ^ string_of_exp e1 ^ ", " ^ string_of_exp e2 ^ ")"
  | Band (e1, e2) -> benclose e1 ^ " AND " ^ benclose e2
  | Bor (e1, e2) -> benclose e1 ^ " OR " ^ benclose e2
  | Bnot e -> "NOT(" ^ string_of_bexp e ^ ")"

let string_of_stmt s =
  match s with
    Var (v, n) -> v ^ " : BITVECTOR(" ^ string_of_int n ^ ");"
  | Array (v, n, m) -> v ^ " : ARRAY BITVECTOR(" ^ string_of_int n ^ ") OF BITVECTOR(" ^ string_of_int m ^ ");"
  | Assert e -> "ASSERT(" ^ string_of_bexp e ^ ");"
  | Query e -> "PUSH;\nQUERY(" ^ string_of_bexp e ^ ");\nPOP;"
  | Comment c -> "% " ^ c
  | Empty -> ""

let string_of_prog p =
  String.concat "\n" (List.map string_of_stmt p) ^ "\n"

let mkv v = V v
let mkconst c = Const c
let mkconstb c = Const (Bin c)
let mkconsth c = Const (Hex c)
let mkconcat e1 e2 = Concat (e1, e2)
let mkextract e i j = Extract (e, i, j)
let mksll e i = Sll (e, i)
let mksrl e i = Srl (e, i)
let mksra e i = Sra (e, i)
let mksextend e i = Sextend (e, i)
let mkuextend e i = Uextend (e, i)
let mknot e = Not e
let mkand e1 e2 = And (e1, e2)
let mkor e1 e2 = Or (e1, e2)
let mkxor e1 e2 = Xor (e1, e2)
let mknand e1 e2 = Nand (e1, e2)
let mknor e1 e2 = Nor (e1, e2)
let mkxnor e1 e2 = Xnor (e1, e2)
let mkneg e = Neg e
let mkadd n e1 e2 = Add (n, e1, e2)
let mksum n es =
  match es with
    [] -> mkconstb (zeros n)
  | hd::[] -> hd
  | hd::tl -> List.fold_left (fun res e -> mkadd n res e) hd tl
let mksub n e1 e2 = Sub (n, e1, e2)
let mkmul n e1 e2 = Mul (n, e1, e2)
let mkdiv n e1 e2 = Div (n, e1, e2)
let mksdiv n e1 e2 = Sdiv (n, e1, e2)
let mkmod n e1 e2 = Mod (n, e1, e2)
let mksmod n e1 e2 = Smod (n, e1, e2)
let mkifte b e1 e2 = Ifte (b, e1, e2)
let mkread e i = Read (e, i)
let mkwrite e i v = Write (e, i, v)

let mkeq e1 e2 = Eq (e1, e2)
let mklt e1 e2 = Lt (e1, e2)
let mkle e1 e2 = Le (e1, e2)
let mkgt e1 e2 = Gt (e1, e2)
let mkge e1 e2 = Ge (e1, e2)
let mkslt e1 e2 = Slt (e1, e2)
let mksle e1 e2 = Sle (e1, e2)
let mksgt e1 e2 = Sgt (e1, e2)
let mksge e1 e2 = Sge (e1, e2)
let mkband e1 e2 = Band (e1, e2)
let mkbands es =
  match es with
    [] -> True
  | hd::[] -> hd
  | hd::tl -> List.fold_left (fun res e -> Band (res, e)) hd tl
let mkbor e1 e2 = Bor (e1, e2)
let mkbors es =
  match es with
    [] -> False
  | hd::[] -> hd
  | hd::tl -> List.fold_left (fun res e -> Bor (res, e)) hd tl
let mkbnot e = Bnot e

let mkvar v n = Var (v, n)
let mkarray v n m = Array (v, n, m)
let mkassert e = Assert e
let mkquery e = Query e
let mkcomment c = Comment c
let mkempty () = Empty
