
open Qhasm

(** Supported formats for bit-vectors. *)
type format = Btor | STP | CVC3 | SMT2

(** Verification conditions represented in supported formats. *)
type vc = 
  VCBtor of Btor.bprog
| VCSTP of Stp.prog
| VCCVC3 of Cvc3.prog
| VCSMT2 of Smt2.prog

let rec split_conj e =
  match e with
    QBexpAnd (e1, e2) -> (split_conj e1)@(split_conj e2)
  | _ -> [e]

let rec mk_conj es =
  match es with
    [] -> QBexpTrue
  | hd::[] -> hd
  | hd::tl -> QBexpAnd (hd, mk_conj tl)

(** 
    * Simplifies a program by removing unnecessary assumptions. 
    * The input program is assumed to contain neither cuts nor invariants.
    * NOTE: If the simplified program is verified, the unsimplified program is verified as well.
    * NOTE: If the simplified program is not verified, the unsimplified program may still be verified.
*)
let over_approximate prog =
  let related stmt vars =
    not (VarSet.is_empty (VarSet.inter (vars_of_qstmt stmt) vars)) in
  let related_assumption vars e =
    (*mk_conj (List.filter (fun e -> not (VarSet.is_empty (VarSet.inter (vars_of_qbexp e) vars))) (split_conj e)) in*)
    mk_conj (List.filter (fun e -> VarSet.subset (vars_of_qbexp e) vars) (split_conj e)) in
  let rec remove_useless prog vars =
    match prog with
      [] -> []
    | hd::tl ->
      let (kindop, vars) =
        match hd.skind with
          QAnnot (QAuxVar (v, eop)) -> 
            if VarSet.mem v vars then
              (Some hd.skind, VarSet.union (vars_of_qstmt hd) vars)
            else
              (None, vars)
        | QAnnot (QConst _) ->
          (Some hd.skind, vars)
        | QAnnot (QFunction fd) ->
          if VarSet.mem fd.svar vars then
            (Some hd.skind, vars)
          else
            (None, vars)
        | QAnnot (QPredicate p) ->
          if VarSet.mem p.pvar vars then
            (Some hd.skind, vars)
          else
            (None, vars)
        | QAnnot (QAssume e) ->
          let related = related_assumption vars e in
          (Some (QAnnot (QAssume related)), VarSet.union (vars_of_qbexp related) vars)
        | QAnnot (QAssert e) ->
          (Some hd.skind, VarSet.union (vars_of_qbexp e) vars)
        | QAnnot a -> failwith ("Unexpected annotation in simplifying Qhasm programs:\n" ^ string_of_qannot a)
        | _ -> 
          if VarSet.is_empty (VarSet.inter (lvals_of_qstmt hd) vars) then
            (Some hd.skind, vars)
          else
            (Some hd.skind, VarSet.union (vars_of_qstmt hd) vars) in
      begin
        match kindop with
          None -> remove_useless tl vars
        | Some kind -> {skind = kind; sline = hd.sline}::(remove_useless tl vars)
      end
  in
  List.rev (remove_useless (List.rev prog) VarSet.empty)

let split_assertion prog =
  let rec helper (res : qprog list) (prog : qprog) (rest : qprog) =
    match rest with
      [] -> res
    | hd::tl ->
      begin
        match hd.skind with
          QAnnot (QAssert p) -> 
            helper (res@(List.map (fun e -> prog@[{skind = QAnnot (QAssert e); sline = hd.sline}]) (split_conj p))) prog tl
        | _ -> helper res (prog@[hd]) tl
      end in
  helper [] [] prog

(** Converts cut to assert and assume and adds invariants to cut. *)
let split_cut prog =
  let rec helper invs res decl prog rest =
    match rest with
      [] -> 
        begin
          match prog with
            [] -> res
          | _ -> res@[prog]
        end
    | hd::tl ->
      begin
        match hd.skind with
        | QVar _ -> helper invs res (decl@[hd]) (prog@[hd]) tl
        (* Preserve function definitions. *)
        | QAnnot (QFunction _) -> helper invs res (decl@[hd]) (prog@[hd]) tl
        (* Preserve predicate definitions. *)
        | QAnnot (QPredicate _) -> helper invs res (decl@[hd]) (prog@[hd]) tl
        (* Convert invariant to assume and remember the invariant. *)
        | QAnnot (QInvariant e) -> 
          let asu = {skind = QAnnot (QAssume e); sline = hd.sline} in
          helper (e::invs) res decl (prog@[asu]) tl
        | QAnnot (QCut p) -> 
          let p = List.fold_left (fun res e -> QBexpAnd (res, e)) p invs in
          let ast = {skind = QAnnot (QAssert p); sline = hd.sline} in
          let asu = {skind = QAnnot (QAssume p); sline = hd.sline} in
          helper invs (res@[prog@[ast]]) decl decl (asu::tl)
        | _ -> helper invs res decl (prog@[hd]) tl
      end in
  let res = helper [] [] [] [] prog in
  res

(** Removes statements after the last assertion. *)
let trim prog =
  let rec helper prog =
    match prog with
      [] -> []
    | hd::tl -> 
      begin
        match hd.skind with
          QAnnot (QAssert _) -> prog
        | _ -> helper tl
      end in
  List.rev (helper (List.rev prog))

(** 
    * Removes statements after the last assertion for each program.
    * Empty programs will be discarded.
*)
let trims progs =
  List.filter (fun prog -> prog <> []) (List.map trim progs)

let split_qprog split_conj approximate prog =
  let progs = split_cut prog in
  let progs = 
    if split_conj then
      List.flatten (List.map split_assertion progs)
    else
      progs in
  let progs = trims progs in
  if approximate then
    List.map over_approximate progs
  else
    progs

(** Generates verification conditions for the current target solver. *)
let generate ?cmap:(cmap=Hashtbl.create 0) ?format:(format=Btor) ?split_conj:(split_conj=false) ?approximate:(approximate=false) p =
  List.map (fun p ->
    match format with
      Btor -> VCBtor (BtorGenerator.generate cmap p)
    | STP -> VCSTP (StpGenerator.generate cmap p)
    | CVC3 -> VCCVC3 (Cvc3Generator.generate cmap p)
    | SMT2 -> VCSMT2 (Smt2Generator.generate cmap p)
  ) (split_qprog split_conj approximate p)

(** Returns the string representations of verification conditions. *)
let string_of_vc p =
  match p with
    VCBtor vc -> Btor.string_of_bprog vc
  | VCSTP vc -> Stp.string_of_prog vc
  | VCCVC3 vc -> Cvc3.string_of_prog vc
  | VCSMT2 vc -> Smt2.string_of_prog vc

(** Returns the suggested file extension. *)
let ext format =
  match format with
    Btor -> ".btor"
  | STP -> ".stp"
  | CVC3 -> ".cvc3"
  | SMT2 -> ".smt2"
