
open Arg
open Qhasm

exception OutputCreationFailure of string

(** The action to be performed. *)
type action = Parse | Generate

(** Generates verification conditions by default. *)
let act = ref Generate

(** A file containing constants. *)
let cfile = ref None

let o_format = ref Generator.Btor

let o_split_conj = ref false

let o_approximate = ref false

(** Suggests a filename for outputting verification conditions. *)
let suggest_filename fn = Filename.chop_suffix (Filename.basename fn) ".q"

(** Saves a verification condition to a file. *)
let save vcg fn =
  let odir = "vc" in
  let _ = 
    try
      if not (Sys.is_directory odir) then
        raise (OutputCreationFailure ("There is a regular file with the same name as our ourput directory: " ^ odir ^ "."))
    with Sys_error _ ->
      Unix.mkdir odir 0o755 in
  let oc = open_out (odir ^ Filename.dir_sep ^ fn) in
  let _ = output_string oc vcg in
  let _ = close_out oc in
  ()

(** Reads constants from a file. *)
let read_consts fn =
  let map = Hashtbl.create 10 in
  let ic = open_in fn in
  let _ =
      try
        while true do
          let line = input_line ic in
          match Str.split (Str.regexp "=") line with
            n::v::[] -> Hashtbl.add map (String.trim n) (String.trim v)
          | _ -> print_endline ("Warning: Failed to read predefined constants from " ^ line ^ ".")
        done
      with End_of_file ->
        () in
  map

let args = 
  [
    ("-a", Set o_approximate, "Over-approximate verification conditions.");
    ("-c", String (fun str -> cfile := Some str), "Specify a file containing predefined constants.");
    ("-o", Symbol (["btor"; "cvc3"; "smt2"; "stp"], 
                   fun str -> 
                     if str = "btor" then
                       o_format := Generator.Btor
                     else if str = "cvc3" then
                       o_format := Generator.CVC3
                     else if str = "smt2" then
                       o_format := Generator.SMT2
                     else if str = "stp" then
                       o_format := Generator.STP
     ), 
     "Specify the output format of verification conditions.");
    ("-p", Unit (fun () -> act := Parse), "Only parse a Qhasm file without generating any verification condition.");
    ("-s", Set o_split_conj, "Split conjunctive assertions.")
  ]

let usage = "qv [-a] [-c CONSTS] [-o FORMAT] [-p] [-s] FILE"

let anon file =
  match !act with
    Parse ->
      let qprog = Parser.qhasm_from_file file in
      print_endline (Qhasm.string_of_qprog qprog)
  | Generate ->
    let _ = print_endline "Parsing Qhasm file." in
    let qprog = Parser.qhasm_from_file file in
    let cmap = 
      match !cfile with
        None -> None
      | Some fn ->
        let _ = print_endline "Reading predefined constants." in
        Some (read_consts fn) in
    let _ = print_endline "Generating verification conditions." in
    let bprog = 
      match cmap with
        None -> Generator.generate ~format:!o_format ~split_conj:!o_split_conj ~approximate:!o_approximate qprog
      | Some cmap -> Generator.generate ~cmap:cmap ~format:!o_format ~split_conj:!o_split_conj ~approximate:!o_approximate qprog in
    let _ = print_endline "Saving verification conditions." in
    let _ = List.fold_left (fun id vcg -> 
      let fn = suggest_filename file ^ "-vc-" ^ string_of_int id ^ (Generator.ext !o_format) in
      let _ = print_endline ("- #" ^ string_of_int id ^ " to " ^ fn) in
      let _ = save vcg fn in
      id + 1
    ) 0 (List.map Generator.string_of_vc bprog) in
    ()

let _ = 
  parse args anon usage
