
open Big_int

exception EvaluationException of string
exception InvalidOperation of string
exception InvalidAnnotation of string

let wordsize = 64

(** a Qhasm variable has a name and a fixed size. *)
type qvar = {
  mutable vname: string;
  mutable vsize: int
}

let mkvar n s = { vname = n; vsize = s }

let carry_var = mkvar "carry" 1

(** a list of Qhasm types for variable declarations. *)
type qtype =
  QInt64
| QInt3232
| QInt6464
| QFloat80
| QStack32
| QStack64
| QStack128
| QStack256
| QStack512

(** a list of Qhasm types for casting. *)
type qtypec =
  QCastInt8
| QCastInt16
| QCastInt32
| QCastInt64
| QCastUInt8
| QCastUInt16
| QCastUInt32
| QCastUInt64

(**
   addr =
   * base + offset
   * base + index
   * base + index * scale            -- the scale is always 8
   * base + offset + index * scale   -- the scale is always 8
*)
type qaddr =
  QAddrBO of qvar * int
| QAddrBI of qvar * qvar
| QAddrBIS of qvar * qvar
| QAddrBOIS of qvar * int * qvar

(**
   * constant
   * var
*)
type qconstvar =
  QIVConst of int
| QIVVar of qvar

(**
   Note: the third case is not valid in Qhasm but is used in the cryptography programs.
   * var
   * *( uint64 * ) (var + constant)
   * *( uint64 * ) &var
*)
type qvarderef =
  QVDVar of qvar
| QVDDeref of qvar * int
| QVDCoef of qvar

(**
   var = expr where expr is
   * constant
   * var
   * carry
   * var + var
   * var + var + constant
   * var + var + var
   * var + var + carry
   * var * constant
   * var * carry
*)
type qexpr =
  QExprConst of int
| QExprVar of qvar
| QExprCarry
| QExprAddVarVar of qvar * qvar
| QExprAddVarVarConst of qvar * qvar * int
| QExprAddVarVarVar of qvar * qvar * qvar
| QExprAddVarVarCarry of qvar * qvar
| QExprMulVarConst of qvar * int
| QExprMulVarCarry of qvar

(**
   var += expr where expr is
   * constant
   * var
   * carry
   * *( uint64 * ) (var + constant)
   * constant + carry
   * var + constant
   * var + carry
   * *( uint64 * ) (var + constant) + carry
   * *( uint64 * ) &var
*)
type qaddexpr =
  QAddExprConst of int
| QAddExprVar of qvar
| QAddExprCarry
| QAddExprDeref of qvar * int
| QAddExprConstCarry of int
| QAddExprVarConst of qvar * int
| QAddExprVarCarry of qvar
| QAddExprDerefCarry of qvar * int
| QAddExprCoef of qvar

(**
   var -= expr where expr is
   * constant
   * var
   * carry
   * var - carry
   * *( uint64 * ) (var + constant)
*)
type qsubexpr = 
  QSubExprConst of int
| QSubExprVar of qvar
| QSubExprCarry
| QSubExprDeref of qvar * int
| QSubExprVarCarry of qvar
| QSubExprDerefCarry of qvar * int

type qfun = {
  svar: qvar;
  sformals: qvar list;
  sexp: qexp
}

(** Predicates *)
and qpred = {
  pvar: qvar;
  pformals: qvar list;
  pbexp: qbexp
}

(**
   * Expressions in annotations.
*)
and qexp =
  QExpConst of big_int
| QExpCarry
| QExpVar of qvarderef
| QExpNeg of qexp
| QExpNot of qexp
| QExpCast of bool * qexp * int
| QExpAdd of qexp * qexp
| QExpSub of qexp * qexp
| QExpMul of qexp * qexp
| QExpAnd of qexp * qexp
| QExpOr of qexp * qexp
| QExpXor of qexp * qexp
| QExpSmod of qexp * qexp
| QExpUmod of qexp * qexp
| QExpPow of qexp * qexp
| QExpConcat of qexp * qexp
| QExpSll of qexp * qexp
| QExpSrl of qexp * qexp
| QExpSra of qexp * qexp
| QExpSlice of qexp * int * int
| QExpApp of qfun * qexp list
| QExpIte of qbexp * qexp * qexp

(**
   * Boolean expressions in annotations.
*)
and qbexp =
  QBexpTrue
| QBexpEq of qexp * qexp
| QBexpNe of qexp * qexp
| QBexpSlt of qexp * qexp
| QBexpSle of qexp * qexp
| QBexpSgt of qexp * qexp
| QBexpSge of qexp * qexp
| QBexpUlt of qexp * qexp
| QBexpUle of qexp * qexp
| QBexpUgt of qexp * qexp
| QBexpUge of qexp * qexp
| QBexpNeg of qbexp
| QBexpAnd of qbexp * qbexp
| QBexpOr of qbexp * qbexp
| QBexpImp of qbexp * qbexp
| QBexpApp of qpred * qexp list

(**
   * Annotations.
*)
type qannot = 
  QAuxVar of qvar * qexp option
| QConst of qexp
| QFunction of qfun
| QPredicate of qpred
| QInvariant of qbexp
| QAssume of qbexp
| QAssert of qbexp
| QCut of qbexp

type qstmtkind =
  QVar of qtype * qvar                                     (** type var *)
| QLoad of qvar * qtypec * qaddr                           (** var = *( type * ) (address) *)
| QStore of qtypec * qaddr * qconstvar                     (** *( type * ) (var + const) = var, *( type * ) (var + const) = const, *( type * ) (var + var) = var, we allow more than Qhasm *)
| QAssign of qvar * qexpr                                  (** var = expr *)
| QAssignIfCarry of qvar * qexpr * bool                    (** var = expr if carry, var = expr if !carry *)
| QCoef of qvar * qvar                                     (** var = *( uint64 * ) &var, used in the cryptography programs *)
| QAdd of qvar * qaddexpr                                  (** var += expr *)
| QAddCarry of qvar * qaddexpr                             (** carry ? var += expr *)
| QSub of qvar * qsubexpr                                  (** var -= expr *)
| QSubCarry of qvar * qsubexpr                             (** carry ? var -= expr *)
| QMul of qvar * qconstvar                                 (** var *= expr *)
| QAnd of qvar * qvarderef                                 (** var &= expr *)
| QOr of qvar * qvarderef                                  (** var |= expr *)
| QXor of qvar * qvarderef                                 (** var ^= expr *)
| QConcatMul of bool * qvar * qvar * qvarderef             (** (int128) t r = r * s, (uint128) t r = r * s, the first argument is true for int128, the last argument should be a variable but dereferences are used in the cryptography programs *)
| QNeg of qvar                                             (** r = -r, two's complement *)
| QNot of qvar                                             (** r = ~r, one's complement *)
| QConcatShiftLeft of qvar * qvar * qconstvar              (** r = (r.t) << s, r = (r.t) << n *)
| QShiftLeft of qvar * qconstvar                           (** r <<= s, r <<= n *)
| QConcatShiftRight of qvar * qvar * qconstvar             (** r = (t r) >> s, r = (t r) >> n *)
| QShiftRight of bool * qvar * qconstvar                   (** (int64) r >>= s, (uint64) r >>= s, (int64) r >>= n, (uint64) r >>= n, the first argument is true if signed *)
| QInput of qvar                                           (** input var, not allowed by Qhasm but used in the cryptography programs *)
| QCaller of qvar                                          (** caller var *)
| QEnter of qvar                                           (** enter name *)
| QLeave                                                   (** leave *)
| QComment of string                                       (** comments *)
| QAnnot of qannot                                         (** annotations *)

type qstmt = {
  (** the kind of statement *)
  skind: qstmtkind;

  (** the line number *)
  sline: int
}

type qprog = qstmt list



let string_of_qtype t =
  match t with
  | QInt64 -> "int64"
  | QInt3232 -> "int3232"
  | QInt6464 -> "int6464"
  | QFloat80 -> "float80"
  | QStack32 -> "stack32"
  | QStack64 -> "stack64"
  | QStack128 -> "stack128"
  | QStack256 -> "stack256"
  | QStack512 -> "stack512"

let string_of_qtypec t =
  match t with
    QCastInt8 -> "int8"
  | QCastInt16 -> "int16"
  | QCastInt32 -> "int32"
  | QCastInt64 -> "int64"
  | QCastUInt8 -> "uint8"
  | QCastUInt16 -> "uint16"
  | QCastUInt32 -> "uint32"
  | QCastUInt64 -> "uint64"

let string_of_qaddr addr =
  match addr with
    QAddrBO (base, offset) -> base.vname ^ " + " ^ string_of_int offset
  | QAddrBI (base, index) -> base.vname ^ " + " ^ index.vname
  | QAddrBIS (base, index) -> base.vname ^ " + " ^ index.vname ^ " * 8"
  | QAddrBOIS (base, offset, index) -> base.vname ^ " + " ^ string_of_int offset ^ " + " ^ index.vname ^ " * 8"

let string_of_qconstvar e =
  match e with
    QIVConst n -> string_of_int n
  | QIVVar v -> v.vname

let string_of_qvarderef e =
  match e with
    QVDVar v -> v.vname
  | QVDDeref (v, off) -> "*(uint64 *)(" ^ v.vname ^ " + " ^ string_of_int off ^ ")"
  | QVDCoef v -> "*(uint64 *) &" ^ v.vname

let string_of_qexpr e =
  match e with
    QExprConst n -> string_of_int n
  | QExprVar v -> v.vname
  | QExprCarry -> "carry"
  | QExprAddVarVar (v1, v2) -> v1.vname ^ " + " ^ v2.vname
  | QExprAddVarVarConst (v1, v2, n) -> v1.vname ^ " + " ^ v2.vname ^ " + " ^ string_of_int n
  | QExprAddVarVarVar (v1, v2, v3) -> v1.vname ^ " + " ^ v2.vname ^ " + " ^ v3.vname
  | QExprAddVarVarCarry (v1, v2) -> v1.vname ^ " + " ^ v2.vname ^ " + carry"
  | QExprMulVarConst (v, n) -> v.vname ^ " * " ^ string_of_int n
  | QExprMulVarCarry v -> v.vname ^ " * carry"

let string_of_qaddexpr e =
  match e with
    QAddExprConst n -> string_of_int n
  | QAddExprVar v -> v.vname
  | QAddExprCarry -> "carry"
  | QAddExprDeref (v, off) -> "*(uint64 *)(" ^ v.vname ^ " + " ^ string_of_int off ^ ")"
  | QAddExprConstCarry n -> string_of_int n ^ " + carry"
  | QAddExprVarConst (v, n) -> v.vname ^ " + " ^ string_of_int n
  | QAddExprVarCarry v -> v.vname ^ " + carry"
  | QAddExprDerefCarry (v, off) -> "*(uint64 *)(" ^ v.vname ^ " + " ^ string_of_int off ^ ") + carry"
  | QAddExprCoef v -> "*(uint64 *) &" ^ v.vname

(**
   var -= expr where expr is
   * constant
   * var
   * carry
   * var - carry
   * *( uint64 * ) (var + constant)
*)
let string_of_qsubexpr e = 
  match e with
    QSubExprConst n -> string_of_int n
  | QSubExprVar v -> v.vname
  | QSubExprCarry -> "carry"
  | QSubExprDeref (v, off) -> "*(uint64 *)(" ^ v.vname ^ " + " ^ string_of_int off ^ ")"
  | QSubExprVarCarry v -> v.vname ^ " - " ^ "carry"
  | QSubExprDerefCarry (v, off) -> "*(uint64 *)(" ^ v.vname ^ " - " ^ string_of_int off ^ ") + carry"

(**
   * Expressions in annotations.
*)
let rec string_of_qexp e =
  match e with
    QExpConst n -> string_of_big_int n
  | QExpCarry -> "carry"
  | QExpVar vd -> string_of_qvarderef vd
  | QExpNeg e -> "-" ^ enclose e
  | QExpNot e -> "~" ^ enclose e
  | QExpCast (signed, e, n) -> enclose e ^ "@" ^ (if signed then "" else "u") ^ string_of_int n
  | QExpAdd (e1, e2) -> enclose e1 ^ " + " ^ enclose e2
  | QExpSub (e1, e2) -> enclose e1 ^ " - " ^ enclose e2
  | QExpMul (e1, e2) -> enclose e1 ^ " * " ^ enclose e2
  | QExpAnd (e1, e2) -> enclose e1 ^ " & " ^ enclose e2
  | QExpOr (e1, e2) -> enclose e1 ^ " | " ^ enclose e2
  | QExpXor (e1, e2) -> enclose e1 ^ " ^ " ^ enclose e2
  | QExpSmod (e1, e2) -> enclose e1 ^ " % " ^ enclose e2
  | QExpUmod (e1, e2) -> enclose e1 ^ " %u " ^ enclose e2
  | QExpPow (e, n) -> enclose e ^ " ** " ^ enclose n
  | QExpConcat (e1, e2) -> enclose e1 ^ "." ^ enclose e2
  | QExpSll(e1, e2) -> enclose e1 ^ " << " ^ enclose e2
  | QExpSrl (e1, e2) -> enclose e1 ^ " >> " ^ enclose e2
  | QExpSra (e1, e2) -> enclose e1 ^ " >>a " ^ enclose e2
  | QExpSlice (e, i, j) -> enclose e ^ "[" ^ string_of_int i ^ ", " ^ string_of_int j ^ "]"
  | QExpApp (fd, actuals) -> fd.svar.vname ^ "(" ^ String.concat ", " (List.map string_of_qexp actuals) ^ ")"
  | QExpIte (b, e1, e2) -> enclose_b b ^ " ? " ^ enclose e1 ^ " : " ^ enclose e2
and enclose e =
  match e with
    QExpConst _
  | QExpCarry
  | QExpVar _ 
  | QExpCast _ -> string_of_qexp e
  | _ -> "(" ^ string_of_qexp e ^ ")"    

(**
   * Boolean expressions in annotations.
*)
and string_of_qbexp be =
  match be with
    QBexpTrue -> "true"
  | QBexpEq (e1, e2) -> string_of_qexp e1 ^ " = " ^ string_of_qexp e2
  | QBexpNe (e1, e2) -> string_of_qexp e1 ^ " != " ^ string_of_qexp e2
  | QBexpSlt (e1, e2) -> string_of_qexp e1 ^ " < " ^ string_of_qexp e2
  | QBexpSle (e1, e2) -> string_of_qexp e1 ^ " <= " ^ string_of_qexp e2
  | QBexpSgt (e1, e2) -> string_of_qexp e1 ^ " > " ^ string_of_qexp e2
  | QBexpSge (e1, e2) -> string_of_qexp e1 ^ " >= " ^ string_of_qexp e2
  | QBexpUlt (e1, e2) -> string_of_qexp e1 ^ " <u " ^ string_of_qexp e2
  | QBexpUle (e1, e2) -> string_of_qexp e1 ^ " <=u " ^ string_of_qexp e2
  | QBexpUgt (e1, e2) -> string_of_qexp e1 ^ " >u " ^ string_of_qexp e2
  | QBexpUge (e1, e2) -> string_of_qexp e1 ^ " >=u " ^ string_of_qexp e2
  | QBexpNeg e -> "~ " ^ enclose_b e
  | QBexpAnd (e1, e2) -> enclose_b e1 ^ " /\\ " ^ enclose_b e2
  | QBexpOr (e1, e2) -> enclose_b e1 ^ " \\/ " ^ enclose_b e2
  | QBexpImp (e1, e2) -> enclose_b e1 ^ " -> " ^ enclose_b e2
  | QBexpApp (p, actuals) -> p.pvar.vname ^ "(" ^ String.concat ", " (List.map string_of_qexp actuals) ^ ")"
and enclose_b e =
  match e with
  | QBexpAnd _
  | QBexpOr _
  | QBexpImp _ -> "(" ^ string_of_qbexp e ^ ")"
  | _ -> string_of_qbexp e

(**
   * Annotations.
*)
let string_of_qannot annot =
  match annot with
    QAuxVar (v, eop) ->
      begin
        match eop with
          None -> v.vname
        | Some e -> v.vname ^ " = " ^ string_of_qexp e
      end
  | QConst e -> string_of_qexp e
  | QFunction fd -> fd.svar.vname ^ "(" ^ (String.concat ", " (List.map (fun v -> v.vname) fd.sformals)) ^ ") = " ^ string_of_qexp fd.sexp
  | QPredicate p -> p.pvar.vname ^ "(" ^ (String.concat ", " (List.map (fun v -> v.vname) p.pformals)) ^ ") = " ^ string_of_qbexp p.pbexp
  | QInvariant e -> "inv " ^ string_of_qbexp e
  | QAssume e -> "assume " ^ string_of_qbexp e
  | QAssert e -> "assert " ^ string_of_qbexp e
  | QCut e -> "cut " ^ string_of_qbexp e

let string_of_qstmtkind k =
  match k with
    QVar (t, v) -> string_of_qtype t ^ " " ^ v.vname
  | QLoad (v, t, addr) -> v.vname ^ " = *(" ^ string_of_qtypec t ^ " *)(" ^ string_of_qaddr addr ^ ")"
  | QStore (t, addr, v) -> "*(" ^ string_of_qtypec t ^ " *)(" ^ string_of_qaddr addr ^ ") = " ^ string_of_qconstvar v
  | QAssign (v, e) -> v.vname ^ " = " ^ string_of_qexpr e
  | QAssignIfCarry (v, e, negative) -> v.vname ^ " = " ^ string_of_qexpr e ^ " if " ^ (if negative then "!" else "") ^ "carry"
  | QCoef (v1, v2) -> v1.vname ^ " = *(uint64 *) &" ^ v2.vname
  | QAdd (v, e) -> v.vname ^ " += " ^ string_of_qaddexpr e
  | QAddCarry (v, e) -> "carry ? " ^ v.vname ^ " += " ^ string_of_qaddexpr e
  | QSub (v, e) -> v.vname ^ " -= " ^ string_of_qsubexpr e
  | QSubCarry (v, e) -> "carry ? " ^ v.vname ^ " -= " ^ string_of_qsubexpr e
  | QMul (v, e) -> v.vname ^ " *= " ^ string_of_qconstvar e
  | QAnd (v, e) -> v.vname ^ " &= " ^ string_of_qvarderef e
  | QOr (v, e) -> v.vname ^ " |= " ^ string_of_qvarderef e
  | QXor (v, e) -> v.vname ^ " ^= " ^ string_of_qvarderef e
  | QConcatMul (s, v1, v2, v3) -> "(" ^ (if s then "int128" else "uint128") ^ ") " ^ v1.vname ^ " " ^ v2.vname ^ " = " ^ v2.vname ^ " * " ^ string_of_qvarderef v3
  | QNeg v -> v.vname ^ " = -" ^ v.vname
  | QNot v -> v.vname ^ " = ~" ^ v.vname
  | QConcatShiftLeft (v1, v2, e) -> v1.vname ^ " = (" ^ v1.vname ^ "." ^ v2.vname ^ ") << " ^ string_of_qconstvar e
  | QShiftLeft (v, e) -> v.vname ^ " <<= " ^ string_of_qconstvar e
  | QConcatShiftRight (v1, v2, e) -> v2.vname ^ " = (" ^ v1.vname ^ " " ^ v2.vname ^ ") >> " ^ string_of_qconstvar e
  | QShiftRight (s, v, e) -> "(" ^ (if s then "int64" else "uint64") ^ ") " ^ v.vname ^ " >>= " ^ string_of_qconstvar e
  | QInput v -> "input " ^ v.vname
  | QCaller v -> "caller " ^ v.vname
  | QEnter v -> "enter " ^ v.vname
  | QLeave -> "leave"
  | QComment str -> "#" ^ str
  | QAnnot annot -> "#// " ^ string_of_qannot annot

let string_of_qstmt stmt = string_of_qstmtkind stmt.skind

let string_of_qprog prog = String.concat "\n" (List.map string_of_qstmt prog)

module VarElm =
struct
  type t = qvar
  let compare v1 v2 = Pervasives.compare v1.vname v2.vname
end

module VarSet = Set.Make(VarElm)

let vars_of_qaddr addr =
  match addr with
    QAddrBO (base, _) -> VarSet.singleton base
  | QAddrBI (base, index)
  | QAddrBIS (base, index)
  | QAddrBOIS (base, _, index) -> VarSet.add base (VarSet.singleton index)

let vars_of_qconstvar e =
  match e with
    QIVConst n -> VarSet.empty
  | QIVVar v -> VarSet.singleton v

let vars_of_qvarderef vd =
  match vd with
    QVDVar v
  | QVDDeref (v, _)
  | QVDCoef v -> VarSet.singleton v

let vars_of_qexpr e =
  match e with
    QExprConst n -> VarSet.empty
  | QExprVar v -> VarSet.singleton v
  | QExprCarry -> VarSet.singleton carry_var
  | QExprAddVarVar (v1, v2)
  | QExprAddVarVarConst (v1, v2, _) -> VarSet.add v1 (VarSet.singleton v2)
  | QExprAddVarVarVar (v1, v2, v3) -> VarSet.add v1 (VarSet.add v2 (VarSet.singleton v3))
  | QExprAddVarVarCarry (v1, v2) -> VarSet.add v1 (VarSet.add v2 (VarSet.singleton carry_var))
  | QExprMulVarConst (v, n) -> VarSet.singleton v
  | QExprMulVarCarry v -> VarSet.add v (VarSet.singleton carry_var)

let vars_of_qaddexpr e =
  match e with
    QAddExprConst n -> VarSet.empty
  | QAddExprVar v -> VarSet.singleton v
  | QAddExprCarry -> VarSet.singleton carry_var
  | QAddExprDeref (v, _) -> VarSet.singleton v
  | QAddExprConstCarry n -> VarSet.singleton carry_var
  | QAddExprVarConst (v, _) -> VarSet.singleton v
  | QAddExprVarCarry v
  | QAddExprDerefCarry (v, _) -> VarSet.add v (VarSet.singleton carry_var)
  | QAddExprCoef v -> VarSet.singleton v

let vars_of_qsubexpr e = 
  match e with
    QSubExprConst n -> VarSet.empty
  | QSubExprVar v -> VarSet.singleton v
  | QSubExprCarry -> VarSet.singleton carry_var
  | QSubExprDeref (v, _) -> VarSet.singleton v
  | QSubExprVarCarry v -> VarSet.add v (VarSet.singleton carry_var)
  | QSubExprDerefCarry (v, _) -> VarSet.add v (VarSet.singleton carry_var)

(** Returns the variables in an expression. *)
let rec vars_of_qexp e =
  match e with
    QExpConst n -> VarSet.empty
  | QExpCarry -> VarSet.singleton carry_var
  | QExpVar vd -> vars_of_qvarderef vd
  | QExpNeg e
  | QExpNot e
  | QExpCast (_, e, _) -> vars_of_qexp e
  | QExpAdd (e1, e2)
  | QExpSub (e1, e2)
  | QExpMul (e1, e2)
  | QExpAnd (e1, e2)
  | QExpOr (e1, e2)
  | QExpXor (e1, e2)
  | QExpSmod (e1, e2)
  | QExpUmod (e1, e2)
  | QExpPow (e1, e2)
  | QExpConcat (e1, e2)
  | QExpSll(e1, e2)
  | QExpSrl (e1, e2)
  | QExpSra (e1, e2) -> VarSet.union (vars_of_qexp e1) (vars_of_qexp e2)
  | QExpSlice (e, _, _) -> vars_of_qexp e
  | QExpApp (fd, actuals) ->
    let vars = List.fold_left (fun vars formal -> VarSet.remove formal vars) (vars_of_qexp fd.sexp) fd.sformals in
    List.fold_left (fun vars actual -> VarSet.union vars (vars_of_qexp actual)) vars actuals
  | QExpIte (b, e1, e2) -> VarSet.union (VarSet.union (vars_of_qbexp b) (vars_of_qexp e1)) (vars_of_qexp e2)

(** Returns the variables in a Boolean expression. *)
and vars_of_qbexp e =
  match e with
    QBexpTrue -> VarSet.empty
  | QBexpEq (e1, e2) -> VarSet.union (vars_of_qexp e1) (vars_of_qexp e2)
  | QBexpNe (e1, e2)
  | QBexpSlt (e1, e2)
  | QBexpSle (e1, e2)
  | QBexpSgt (e1, e2)
  | QBexpSge (e1, e2)
  | QBexpUlt (e1, e2)
  | QBexpUle (e1, e2)
  | QBexpUgt (e1, e2)
  | QBexpUge (e1, e2) -> VarSet.union (vars_of_qexp e1) (vars_of_qexp e2)
  | QBexpNeg e -> vars_of_qbexp e
  | QBexpAnd (e1, e2)
  | QBexpOr (e1, e2)
  | QBexpImp (e1, e2) -> VarSet.union (vars_of_qbexp e1) (vars_of_qbexp e2)
  | QBexpApp (p, actuals) ->
    let vars = List.fold_left (fun vars formal -> VarSet.remove formal vars) (vars_of_qbexp p.pbexp) p.pformals in
    List.fold_left (fun vars actual -> VarSet.union vars (vars_of_qexp actual)) vars actuals

let vars_of_qannot annot =
  match annot with
    QAuxVar (v, eop) ->
      begin
        match eop with
          None -> VarSet.singleton v
        | Some e -> VarSet.add v (vars_of_qexp e)
      end
  | QConst e -> vars_of_qexp e
  | QFunction fd -> VarSet.singleton fd.svar
  | QPredicate p -> VarSet.singleton p.pvar
  | QInvariant e
  | QAssume e
  | QAssert e
  | QCut e -> vars_of_qbexp e

let vars_of_qstmtkind k =
  match k with
    QVar (_, v) -> VarSet.singleton v
  | QLoad (v, _, addr) -> VarSet.add v (vars_of_qaddr addr)
  | QStore (_, addr, v) -> VarSet.union (vars_of_qaddr addr) (vars_of_qconstvar v)
  | QAssign (v, e) -> VarSet.add v (vars_of_qexpr e)
  | QAssignIfCarry (v, e, _) -> VarSet.add carry_var (VarSet.add v (vars_of_qexpr e))
  | QCoef (v1, v2) -> VarSet.add v1 (VarSet.singleton v2)
  | QAdd (v, e) -> VarSet.add v (vars_of_qaddexpr e)
  | QAddCarry (v, e) -> VarSet.add v (VarSet.add carry_var (vars_of_qaddexpr e))
  | QSub (v, e) -> VarSet.add v (vars_of_qsubexpr e)
  | QSubCarry (v, e) -> VarSet.add v (VarSet.add carry_var (vars_of_qsubexpr e))
  | QMul (v, e) -> VarSet.add v (vars_of_qconstvar e)
  | QAnd (v, e)
  | QOr (v, e)
  | QXor (v, e) -> VarSet.add v (vars_of_qvarderef e)
  | QConcatMul (s, v1, v2, v3) -> VarSet.add v1 (VarSet.add v2 (vars_of_qvarderef v3))
  | QNeg v
  | QNot v -> VarSet.singleton v
  | QConcatShiftLeft (v1, v2, e) -> VarSet.add v1 (VarSet.add v2 (vars_of_qconstvar e))
  | QShiftLeft (v, e) -> VarSet.add v (vars_of_qconstvar e)
  | QConcatShiftRight (v1, v2, e) -> VarSet.add v1 (VarSet.add v2 (vars_of_qconstvar e))
  | QShiftRight (s, v, e) -> VarSet.add v (vars_of_qconstvar e)
  | QInput _
  | QCaller _
  | QEnter _
  | QLeave
  | QComment _ -> VarSet.empty
  | QAnnot annot -> vars_of_qannot annot

let vars_of_qstmt stmt = vars_of_qstmtkind stmt.skind

let vars_of_qprog prog = List.fold_left (fun vars stmt -> VarSet.union vars (vars_of_qstmt stmt)) VarSet.empty prog

let lvals_of_qstmtkind k =
  match k with
    QVar (_, v)
  | QLoad (v, _, _) -> VarSet.singleton v
  | QStore (_, addr, v) -> vars_of_qaddr addr
  | QAssign (v, _)
  | QAssignIfCarry (v, _, _)
  | QCoef (v, _)
  | QAdd (v, _)
  | QAddCarry (v, _)
  | QSub (v, _)
  | QSubCarry (v, _)
  | QMul (v, _)
  | QAnd (v, _)
  | QOr (v, _)
  | QXor (v, _) -> VarSet.singleton v
  | QConcatMul (_, v1, v2, _) -> VarSet.add v1 (VarSet.singleton v2)
  | QNeg v
  | QNot v -> VarSet.singleton v
  | QConcatShiftLeft (v, _, _)
  | QShiftLeft (v, _)
  | QConcatShiftRight (_, v, _)
  | QShiftRight (_, v, _) -> VarSet.singleton v
  | QInput _
  | QCaller _
  | QEnter _
  | QLeave
  | QComment _ -> VarSet.empty
  | QAnnot annot -> VarSet.empty

let lvals_of_qstmt stmt = lvals_of_qstmtkind stmt.skind

let size_of_qtype qt =
  match qt with
    QInt64 -> 64
  | QInt3232 -> 64
  | QInt6464 -> 128
  | QFloat80 -> 80
  | QStack32 -> 32
  | QStack64 -> 64
  | QStack128 -> 128
  | QStack256 -> 256
  | QStack512 -> 512

let size_of_qtypec qt =
  match qt with
    QCastInt8
  | QCastUInt8 -> 8
  | QCastInt16
  | QCastUInt16 -> 16
  | QCastInt32
  | QCastUInt32 -> 32
  | QCastInt64
  | QCastUInt64 -> 64

let signed qt =
  match qt with
    QCastInt8
  | QCastInt16
  | QCastInt32
  | QCastInt64 -> true
  | QCastUInt8
  | QCastUInt16
  | QCastUInt32
  | QCastUInt64 -> false

(** 
    * Returns true if an expression is pure. An expression is pure if it has
    * neither variable, carry, type casting, concatenation, nor slicing.
*)
let rec pure e =
  match e with
    QExpConst _ -> true
  | QExpCarry
  | QExpVar _
  | QExpCast _ -> false
  | QExpNeg e
  | QExpNot e -> pure e
  | QExpAdd (e1, e2)
  | QExpSub (e1, e2)
  | QExpMul (e1, e2)
  | QExpAnd (e1, e2)
  | QExpOr (e1, e2)
  | QExpXor (e1, e2)
  | QExpSmod (e1, e2)
  | QExpUmod (e1, e2) -> pure e1 && pure e2
  | QExpPow (e, n) -> pure e && pure n
  | QExpConcat _ -> false
  | QExpSll (e1, e2)
  | QExpSrl (e1, e2)
  | QExpSra (e1, e2) -> pure e1 && pure e2
  | QExpSlice (e, _, _) -> false
  | QExpApp _ -> false
  | QExpIte (b, e1, e2) -> bpure b && pure e1 && pure e2

(** Returns true if a Boolean expression is pure. *)
and bpure e =
  match e with
    QBexpTrue -> true
  | QBexpEq (e1, e2) -> pure e1 && pure e2
  | QBexpNe (e1, e2)
  | QBexpSlt (e1, e2)
  | QBexpSle (e1, e2)
  | QBexpSgt (e1, e2)
  | QBexpSge (e1, e2)
  | QBexpUlt (e1, e2)
  | QBexpUle (e1, e2)
  | QBexpUgt (e1, e2)
  | QBexpUge (e1, e2) -> pure e1 && pure e2
  | QBexpNeg e -> bpure e
  | QBexpAnd (e1, e2)
  | QBexpOr (e1, e2)
  | QBexpImp (e1, e2) -> bpure e1 && bpure e2
  | QBexpApp _ -> false

(** Evaluates a pure expression. *)
let rec eval e =
  match e with
    QExpConst n -> n
  | QExpCarry -> raise (EvaluationException ("Carry cannot be evaluated."))
  | QExpVar vd -> raise (EvaluationException ("Variable " ^ string_of_qvarderef vd ^ " cannot be evaluated."))
  | QExpNeg e -> minus_big_int (eval e)
  | QExpNot e -> raise (EvaluationException ("Bit-wise not of pure expressions is not supported."))
  | QExpCast _ -> raise (EvaluationException ("Type casting cannot be evaluated."))
  | QExpAdd (e1, e2) -> add_big_int (eval e1) (eval e2)
  | QExpSub (e1, e2) -> sub_big_int (eval e1) (eval e2)
  | QExpMul (e1, e2) -> mult_big_int (eval e1) (eval e2)
  | QExpAnd (e1, e2) -> and_big_int (eval e1) (eval e2)
  | QExpOr (e1, e2) -> or_big_int (eval e1) (eval e2)
  | QExpXor (e1, e2) -> xor_big_int (eval e1) (eval e2)
  | QExpSmod (e1, e2) -> mod_big_int (eval e1) (eval e2)
  | QExpUmod (e1, e2) -> mod_big_int (eval e1) (eval e2)
  | QExpPow (e, n) -> power_big_int_positive_big_int (eval e) (eval n)
  | QExpConcat _ -> raise (EvaluationException ("Concatenation cannot be evaluated."))
  | QExpSll (e1, e2) -> shift_left_big_int (eval e1) (int_of_big_int (eval e2))
  | QExpSrl (e1, e2) -> shift_right_towards_zero_big_int (eval e1) (int_of_big_int (eval e2))
  | QExpSra (e1, e2) -> shift_right_big_int (eval e1) (int_of_big_int (eval e2)) 
  | QExpSlice _ -> raise (EvaluationException ("Slicing cannot be evaluated."))
  | QExpApp _ -> raise (EvaluationException ("Function application cannot be evaluated."))
  | QExpIte _ -> raise (EvaluationException ("If-then-else cannot be evaluated."))

(** Evaluates a pure expression as an integer. Raise EvaluationException if the expression cannot be expressed as an integer. *)
let rec eval_int e =
  let n = eval e in
  if is_int_big_int n then
    int_of_big_int n
  else
    raise (EvaluationException (string_of_qexp e ^ " cannot be evaluated as an integer."))

(**
   * Returns the estimated size of an expression. The wordsize is always returned
   * for constants and carry.
*)
let rec size_of_exp exp =
  match exp with
    QExpConst n -> wordsize
  | QExpCarry -> 1
  | QExpVar vd -> 
    begin
      match vd with
        QVDVar v -> v.vsize
      | QVDDeref _ -> wordsize
      | QVDCoef _ -> wordsize
    end
  | QExpNeg e
  | QExpNot e -> size_of_exp e
  | QExpCast (signed, e, n) -> n
  | QExpAdd (e1, e2)
  | QExpSub (e1, e2)
  | QExpMul (e1, e2)
  | QExpAnd (e1, e2)
  | QExpOr (e1, e2)
  | QExpXor (e1, e2)
  | QExpSmod (e1, e2) -> max (size_of_exp e1) (size_of_exp e2)
  | QExpUmod (e1, e2) -> (max (size_of_exp e1) (size_of_exp e2) + 1)
  | QExpPow (e, _) -> size_of_exp e
  | QExpConcat (e1, e2) -> (size_of_exp e1) + (size_of_exp e2)
  | QExpSll (e, _)
  | QExpSrl (e, _)
  | QExpSra (e, _) -> size_of_exp e
  | QExpSlice (_, i, j) -> i - j + 1
  | QExpApp (fd, actuals) -> fd.svar.vsize
  | QExpIte (_, e1, e2) -> max (size_of_exp e1) (size_of_exp e2)

(** Substitutes variables in an expression. *)
let rec subst m exp =
  let hash = Hashtbl.create (List.length m) in
  let _ = List.iter (
    fun (formal, actual) ->
      if formal.vsize <> size_of_exp actual then
        raise (InvalidOperation ("The bit-width of the substitution " ^ formal.vname ^ " => " ^ string_of_qexp actual ^ " does not match."))
      else
        Hashtbl.add hash formal.vname actual
  ) m in
  sub hash exp

and subst_b m exp =
  let hash = Hashtbl.create (List.length m) in
  let _ = List.iter (
    fun (formal, actual) ->
      if formal.vsize <> size_of_exp actual then
        raise (InvalidOperation ("The bit-width of the substitution " ^ formal.vname ^ " => " ^ string_of_qexp actual ^ " does not match."))
      else
        Hashtbl.add hash formal.vname actual
  ) m in
  sub_b hash exp

and sub m exp =
  match exp with
    QExpConst n -> exp
  | QExpCarry -> if Hashtbl.mem m "carry" then Hashtbl.find m "carry" else QExpCarry
  | QExpVar varderef ->
    begin
      match varderef with
        QVDVar v -> if Hashtbl.mem m v.vname then Hashtbl.find m v.vname else exp
      | QVDDeref (v, n) ->
        if Hashtbl.mem m v.vname then 
          let sub = Hashtbl.find m v.vname in
          begin
            match sub with
              QExpVar (QVDVar v) -> QExpVar (QVDDeref (v, n))
            | _ -> raise (InvalidOperation "The base address of a dereference can only be replaced by a variable.")
          end
        else 
          exp
      | QVDCoef v -> if Hashtbl.mem m v.vname then Hashtbl.find m v.vname else exp
    end
  | QExpNeg e -> QExpNeg (sub m e)
  | QExpNot e -> QExpNot (sub m e)
  | QExpCast (signed, e, n) -> QExpCast (signed, sub m e, n)
  | QExpAdd (e1, e2) -> QExpAdd (sub m e1, sub m e2)
  | QExpSub (e1, e2) -> QExpSub (sub m e1, sub m e2)
  | QExpMul (e1, e2) -> QExpMul (sub m e1, sub m e2)
  | QExpAnd (e1, e2) -> QExpAnd (sub m e1, sub m e2)
  | QExpOr (e1, e2) -> QExpOr (sub m e1, sub m e2)
  | QExpXor (e1, e2) -> QExpXor (sub m e1, sub m e2)
  | QExpSmod (e1, e2) -> QExpSmod (sub m e1, sub m e2)
  | QExpUmod (e1, e2) -> QExpUmod (sub m e1, sub m e2)
  | QExpPow (e, n) -> QExpPow (sub m e, sub m n)
  | QExpConcat (e1, e2) -> QExpConcat (sub m e1, sub m e2)
  | QExpSll (e1, e2) -> QExpSll (sub m e1, sub m e2)
  | QExpSrl (e1, e2) -> QExpSrl (sub m e1, sub m e2)
  | QExpSra (e1, e2) -> QExpSra (sub m e1, sub m e2)
  | QExpSlice (e, i, j) -> QExpSlice (sub m e, i, j)
  | QExpApp (fn, actuals) -> QExpApp (fn, List.map (sub m) actuals)
  | QExpIte (b, e1, e2) -> QExpIte (sub_b m b, sub m e1, sub m e2)

and sub_b m e =
  match e with
    QBexpTrue -> QBexpTrue
  | QBexpEq (e1, e2) -> QBexpEq (sub m e1, sub m e2)
  | QBexpNe (e1, e2) -> QBexpNe (sub m e1, sub m e2)
  | QBexpSlt (e1, e2) -> QBexpSlt (sub m e1, sub m e2)
  | QBexpSle (e1, e2) -> QBexpSle (sub m e1, sub m e2)
  | QBexpSgt (e1, e2) -> QBexpSgt (sub m e1, sub m e2)
  | QBexpSge (e1, e2) -> QBexpSge (sub m e1, sub m e2)
  | QBexpUlt (e1, e2) -> QBexpUlt (sub m e1, sub m e2)
  | QBexpUle (e1, e2) -> QBexpUle (sub m e1, sub m e2)
  | QBexpUgt (e1, e2) -> QBexpUgt (sub m e1, sub m e2)
  | QBexpUge (e1, e2) -> QBexpUge (sub m e1, sub m e2)
  | QBexpNeg e -> QBexpNeg (sub_b m e)
  | QBexpAnd (e1, e2) -> QBexpAnd (sub_b m e1, sub_b m e2)
  | QBexpOr (e1, e2) -> QBexpOr (sub_b m e1, sub_b m e2)
  | QBexpImp (e1, e2) -> QBexpImp (sub_b m e1, sub_b m e2)
  | QBexpApp (p, actuals) -> QBexpApp (p, List.map (sub m) actuals)

(** 
    * Returns a functor that takes a list of expressions and returns an
    * expression obtained by substituting the expressions for the specified
    * variables in the specified expression.
*)
let mkfunctor e formals =
  fun actuals ->
    try
      subst (List.combine formals actuals) e
    with (Invalid_argument _) ->
        raise (InvalidOperation ("The number of actuals (" ^ string_of_int (List.length actuals) ^ 
                 ") does not match the number of formals (" ^ string_of_int (List.length formals) ^ ")."))

(** 
    * Returns a functor that takes a list of expressions and returns a Boolean
    * expression obtained by substituting the expressions for the specified
    * variables in the specified Boolean expression.
*)
let mkfunctor_b e formals =
  fun actuals ->
    try
      subst_b (List.combine formals actuals) e
    with (Invalid_argument _) ->
        raise (InvalidOperation ("The number of actuals (" ^ string_of_int (List.length actuals) ^ 
                 ") does not match the number of formals (" ^ string_of_int (List.length formals) ^ ")."))
