
open Big_int

exception InvalidAnnotation of string

(** a Qhasm variable has a name and a fixed size. *)
type qvar = {
  mutable vname: string;
  mutable vsize: int
}

(** a list of Qhasm types for variable declarations. *)
type qtype =
  QInt64
| QInt3232
| QInt6464
| QFloat80
| QStack32
| QStack64
| QStack128
| QStack256
| QStack512

(** a list of Qhasm types for casting. *)
type qtypec =
  QCastInt8
| QCastInt16
| QCastInt32
| QCastInt64
| QCastUInt8
| QCastUInt16
| QCastUInt32
| QCastUInt64

(**
   addr =
   * base + offset
   * base + index
   * base + index * scale            -- the scale is always 8
   * base + offset + index * scale   -- the scale is always 8
*)
type qaddr =
  QAddrBO of qvar * int
| QAddrBI of qvar * qvar
| QAddrBIS of qvar * qvar
| QAddrBOIS of qvar * int * qvar

(**
   * constant
   * var
*)
type qconstvar =
  QIVConst of int
| QIVVar of qvar

(**
   * var
   * *( uint64 * ) (var + constant)
*)
type qvarderef =
  QVDVar of qvar
| QVDDeref of qvar * int
| QVDCoef of qvar

(**
   var = expr where expr is
   * constant
   * var
   * carry
   * var + var
   * var + var + constant
   * var + var + var
   * var + var + carry
   * var * constant
   * var * carry
*)
type qexpr =
  QExprConst of int
| QExprVar of qvar
| QExprCarry
| QExprAddVarVar of qvar * qvar
| QExprAddVarVarConst of qvar * qvar * int
| QExprAddVarVarVar of qvar * qvar * qvar
| QExprAddVarVarCarry of qvar * qvar
| QExprMulVarConst of qvar * int
| QExprMulVarCarry of qvar

(**
   var += expr where expr is
   * constant
   * var
   * carry
   * *( uint64 * ) (var + constant)
   * constant + carry
   * var + constant
   * var + carry
   * *( uint64 * ) (var + constant) + carry
*)
type qaddexpr =
  QAddExprConst of int
| QAddExprVar of qvar
| QAddExprCarry
| QAddExprDeref of qvar * int
| QAddExprConstCarry of int
| QAddExprVarConst of qvar * int
| QAddExprVarCarry of qvar
| QAddExprDerefCarry of qvar * int
| QAddExprCoef of qvar

(**
   var -= expr where expr is
   * constant
   * var
   * carry
   * var - carry
   * *( uint64 * ) (var + constant)
*)
type qsubexpr = 
  QSubExprConst of int
| QSubExprVar of qvar
| QSubExprCarry
| QSubExprDeref of qvar * int
| QSubExprVarCarry of qvar
| QSubExprDerefCarry of qvar * int

type qfun = {
  svar: qvar;
  sformals: qvar list;
  sexp: qexp
}

(** Predicates *)
and qpred = {
  pvar: qvar;
  pformals: qvar list;
  pbexp: qbexp
}

(**
   * Expressions in annotations.
*)
and qexp =
  QExpConst of big_int
| QExpCarry
| QExpVar of qvarderef
| QExpNeg of qexp
| QExpNot of qexp
| QExpCast of bool * qexp * int
| QExpAdd of qexp * qexp
| QExpSub of qexp * qexp
| QExpMul of qexp * qexp
| QExpAnd of qexp * qexp
| QExpOr of qexp * qexp
| QExpXor of qexp * qexp
| QExpSmod of qexp * qexp
| QExpUmod of qexp * qexp
| QExpPow of qexp * qexp
| QExpConcat of qexp * qexp
| QExpSll of qexp * qexp
| QExpSrl of qexp * qexp
| QExpSra of qexp * qexp
| QExpSlice of qexp * int * int
| QExpApp of qfun * qexp list
| QExpIte of qbexp * qexp * qexp

(**
   * Boolean expressions in annotations.
*)
and qbexp =
  QBexpTrue
| QBexpEq of qexp * qexp
| QBexpNe of qexp * qexp
| QBexpSlt of qexp * qexp
| QBexpSle of qexp * qexp
| QBexpSgt of qexp * qexp
| QBexpSge of qexp * qexp
| QBexpUlt of qexp * qexp
| QBexpUle of qexp * qexp
| QBexpUgt of qexp * qexp
| QBexpUge of qexp * qexp
| QBexpNeg of qbexp
| QBexpAnd of qbexp * qbexp
| QBexpOr of qbexp * qbexp
| QBexpImp of qbexp * qbexp
| QBexpApp of qpred * qexp list

(**
   * Annotations.
*)
type qannot = 
  QAuxVar of qvar * qexp option
| QConst of qexp
| QFunction of qfun
| QPredicate of qpred
| QInvariant of qbexp
| QAssume of qbexp
| QAssert of qbexp
| QCut of qbexp

type qstmtkind =
  QVar of qtype * qvar                                     (** type var *)
| QLoad of qvar * qtypec * qaddr                           (** var = *( type * ) (address) *)
| QStore of qtypec * qaddr * qconstvar                     (** *( type * ) (var + const) = var, *( type * ) (var + const) = const, *( type * ) (var + var) = var, we allow more than Qhasm *)
| QAssign of qvar * qexpr                                  (** var = expr *)
| QAssignIfCarry of qvar * qexpr * bool                    (** var = expr if carry, var = expr if !carry *)
| QCoef of qvar * qvar                                     (** var = *( uint64 * ) &var, used in the cryptography programs *)
| QAdd of qvar * qaddexpr                                  (** var += expr *)
| QAddCarry of qvar * qaddexpr                             (** carry ? var += expr *)
| QSub of qvar * qsubexpr                                  (** var -= expr *)
| QSubCarry of qvar * qsubexpr                             (** carry ? var -= expr *)
| QMul of qvar * qconstvar                                 (** var *= expr *)
| QAnd of qvar * qvarderef                                 (** var &= expr *)
| QOr of qvar * qvarderef                                  (** var |= expr *)
| QXor of qvar * qvarderef                                 (** var ^= expr *)
| QConcatMul of bool * qvar * qvar * qvarderef             (** (int128) t r = r * s, (uint128) t r = r * s, the first argument is true for int128, the last argument should be a variable but dereferences are used in the cryptography programs *)
| QNeg of qvar                                             (** r = -r, two's complement *)
| QNot of qvar                                             (** r = ~r, one's complement *)
| QConcatShiftLeft of qvar * qvar * qconstvar              (** r = (r.t) << s, r = (r.t) << n *)
| QShiftLeft of qvar * qconstvar                           (** r <<= s, r <<= n *)
| QConcatShiftRight of qvar * qvar * qconstvar             (** r = (t r) >> s, r = (t r) >> n *)
| QShiftRight of bool * qvar * qconstvar                   (** (int64) r >>= s, (uint64) r >>= s, (int64) r >>= n, (uint64) r >>= n, the first argument is true if signed *)
| QInput of qvar                                           (** input var, not allowed by Qhasm but used in the cryptography programs *)
| QCaller of qvar                                          (** caller var *)
| QEnter of qvar                                           (** enter name *)
| QLeave                                                   (** leave *)
| QComment of string                                       (** comments *)
| QAnnot of qannot                                         (** annotations *)

type qstmt = {
  (** the kind of statement *)
  skind: qstmtkind;

  (** the line number *)
  sline: int
}

type qprog = qstmt list

(** Substitutes variables in an expression. *)
val subst : (qvar * qexp) list -> qexp -> qexp

(** 
    * Returns a functor that takes a list of expressions and returns an
    * expression obtained by substituting the expressions for the specified
    * variables in the specified expression.
*)
val mkfunctor : qexp -> qvar list -> (qexp list -> qexp)

(** 
    * Returns a functor that takes a list of expressions and returns a Boolean
    * expression obtained by substituting the expressions for the specified
    * variables in the specified Boolean expression.
*)
val mkfunctor_b : qbexp -> qvar list -> (qexp list -> qbexp)

val string_of_qtype : qtype -> string
val string_of_qtypec : qtypec -> string
val string_of_qaddr : qaddr -> string
val string_of_qconstvar : qconstvar -> string
val string_of_qvarderef : qvarderef -> string
val string_of_qexpr : qexpr -> string
val string_of_qaddexpr : qaddexpr -> string
val string_of_qsubexpr : qsubexpr -> string
val string_of_qexp : qexp -> string
val string_of_qbexp : qbexp -> string
val string_of_qannot : qannot -> string
val string_of_qstmtkind : qstmtkind -> string
val string_of_qstmt : qstmt -> string
val string_of_qprog : qprog -> string

val size_of_qtype : qtype -> int
val size_of_qtypec : qtypec -> int
val signed : qtypec -> bool

val mkvar : string -> int -> qvar

(** 
    * Returns true if an expression is pure. An expression is pure if it has
    * neither variable, carry, type casting, concatenation, nor slicing.
*)
val pure : qexp -> bool

(** Returns true if a Boolean expression is pure. *)
val bpure : qbexp -> bool

(** Evaluates a pure expression. *)
val eval : qexp -> big_int

(** Evaluates a pure expression as an integer. Raise EvaluationException if the expression cannot be expressed as an integer. *)
val eval_int : qexp -> int

(**
   * Returns the estimated size of an expression. The wordsize is always returned
   * for constants and carry.
*)
val size_of_exp : qexp -> int

module VarElm : Set.OrderedType with type t = qvar

module VarSet : Set.S with type elt = VarElm.t

val vars_of_qaddr : qaddr -> VarSet.t
val vars_of_qconstvar : qconstvar -> VarSet.t
val vars_of_qvarderef : qvarderef -> VarSet.t
val vars_of_qexpr : qexpr -> VarSet.t
val vars_of_qaddexpr : qaddexpr -> VarSet.t
val vars_of_qsubexpr : qsubexpr -> VarSet.t
(** Returns the variables in an expression. *)
val vars_of_qexp : qexp -> VarSet.t
(** Returns the variables in a Boolean expression. *)
val vars_of_qbexp : qbexp -> VarSet.t
val vars_of_qannot : qannot -> VarSet.t
val vars_of_qstmtkind : qstmtkind -> VarSet.t
val vars_of_qstmt : qstmt -> VarSet.t
val vars_of_qprog : qprog -> VarSet.t

val lvals_of_qstmtkind : qstmtkind -> VarSet.t
val lvals_of_qstmt : qstmt -> VarSet.t
