(**
   * Z3
*)

let rec zeros n =
  if n = 0 then
    ""
  else
    "0" ^ zeros (n - 1)

(** A variable in Z3. *)
type var = string

(** Constants. *)
type const =
  Bin of string
| Hex of string

(** An expression in Z3. *)
type exp =
  V of var
| Const of const
| Concat of exp * exp
| Extract of exp * int * int
| Sll of exp * exp
| Srl of exp * exp
| Sra of exp * exp
| SignExtend of exp * int
| ZeroExtend of exp * int
| Not of exp
| And of exp * exp
| Or of exp * exp
| Xor of exp * exp
| Nand of exp * exp
| Nor of exp * exp
| Xnor of exp * exp
| Neg of exp
| Add of exp * exp
| Sub of exp * exp
| Mul of exp * exp
| Smod of exp * exp
| Ite of bexp * exp * exp
| Select of exp * exp
| Store of exp * exp * exp

(** A Boolean expression in Z3. *)
and bexp =
  True
| False
| Eq of exp * exp
| Ult of exp * exp
| Ule of exp * exp
| Ugt of exp * exp
| Uge of exp * exp
| Slt of exp * exp
| Sle of exp * exp
| Sgt of exp * exp
| Sge of exp * exp
| Band of bexp * bexp
| Bor of bexp * bexp
| Bnot of bexp

(** Statements in Z3. *)
type stmt = 
  Var of var * int
| Array of var * int * int
| Define of var * int * exp
| Assert of bexp
| Sat of bexp
| Comment of string
| Empty

(** A program in Z3. *)
type prog = stmt list

let rec enclose e =
  match e with
    V _ 
  | Const _ -> string_of_exp e
  | _ -> "(" ^ string_of_exp e ^ ")"

and benclose e =
  match e with
    True
  | False -> string_of_bexp e
  | _ -> "(" ^ string_of_bexp e ^ ")"

and string_of_exp e =
  match e with
    V v -> v
  | Const c -> 
    begin
      match c with 
        Bin n -> "#b" ^ n
      | Hex n -> "#x" ^ n
    end
  | Concat (e1, e2) -> "concat " ^ enclose e1 ^ " " ^ enclose e2
  | Extract (e, i, j) -> "(_ extract " ^ string_of_int i ^ " " ^ string_of_int j ^ ") " ^ enclose e
  | Sll (e, n) -> "bvshl " ^ enclose e ^ " " ^ enclose n
  | Srl (e, n) -> "bvlshr " ^ enclose e ^ " " ^ enclose n
  | Sra (e, n) -> "bvashr " ^ enclose e ^ " " ^ enclose n
  | SignExtend (e, i) -> "(_ sign_extend " ^ string_of_int i ^ ") " ^ enclose e
  | ZeroExtend (e, i) -> "(_ zero_extend " ^ string_of_int i ^ ") " ^ enclose e
  | Not e -> "bvnot " ^ enclose e
  | And (e1, e2) -> "bvand " ^ enclose e1 ^ " " ^ enclose e2
  | Or (e1, e2) -> "bvor " ^ enclose e1 ^ " " ^ enclose e2
  | Xor (e1, e2) -> "bvxor " ^ enclose e1 ^ " " ^ enclose e2
  | Nand (e1, e2) -> "bvnand " ^ enclose e1 ^ " " ^ enclose e2
  | Nor (e1, e2) -> "bvnor " ^ enclose e1 ^ " " ^ enclose e2
  | Xnor (e1, e2) -> "bvxnor " ^ enclose e1 ^ " " ^ enclose e2
  | Neg e -> "bvneg " ^ enclose e
  | Add (e1, e2) -> "bvadd " ^ enclose e1 ^ " " ^ enclose e2
  | Sub (e1, e2) -> "bvsub " ^ enclose e1 ^ " " ^ enclose e2
  | Mul (e1, e2) -> "bvmul " ^ enclose e1 ^ " " ^ enclose e2
  | Smod (e1, e2) -> "bvsmod " ^ enclose e1 ^ " " ^ enclose e2
  | Ite (b, e1, e2) -> "ite " ^ benclose b ^ " " ^ enclose e1 ^ " " ^ enclose e2
  | Select (e, i) -> "select " ^ enclose e ^ " " ^ enclose i
  | Store (e, i, v) -> "store " ^ enclose e ^ " " ^ enclose i ^ " " ^ enclose v

and string_of_bexp e =
  match e with
    True -> "true"
  | False -> "false"
  | Eq (e1, e2) -> "= " ^ enclose e1 ^ " " ^ enclose e2
  | Ult (e1, e2) -> "bvult " ^ enclose e1 ^ " " ^ enclose e2
  | Ule (e1, e2) -> "bvule " ^ enclose e1 ^ " " ^ enclose e2
  | Ugt (e1, e2) -> "bvugt " ^ enclose e1 ^ " " ^ enclose e2
  | Uge (e1, e2) -> "bvuge " ^ enclose e1 ^ " " ^ enclose e2
  | Slt (e1, e2) -> "bvslt " ^ enclose e1 ^ " " ^ enclose e2
  | Sle (e1, e2) -> "bvsle " ^ enclose e1 ^ " " ^ enclose e2
  | Sgt (e1, e2) -> "bvsgt " ^ enclose e1 ^ " " ^ enclose e2
  | Sge (e1, e2) -> "bvsge " ^ enclose e1 ^ " " ^ enclose e2
  | Band (e1, e2) -> "and " ^ benclose e1 ^ " " ^ benclose e2
  | Bor (e1, e2) -> "or " ^ benclose e1 ^ " " ^ benclose e2
  | Bnot e -> "not " ^ benclose e

let string_of_stmt s =
  match s with
    Var (v, n) -> "(declare-fun " ^ v ^ " () (_ BitVec " ^ string_of_int n ^ "))"
  | Array (v, n, m) -> "(declare-fun " ^ v ^ " () (Array (_ BitVec " ^ string_of_int n ^ ") (_ BitVec " ^ string_of_int m ^ ")))"
  | Define (v, n, e) -> "(define-fun " ^ v ^ " () (_ BitVec " ^ string_of_int n ^ ") " ^ enclose e ^ ")"
  | Assert e -> "(assert " ^ benclose e ^ ")"
  | Sat e -> "(push)\n(assert " ^ benclose e ^ ")\n(check-sat)\n(pop)"
  | Comment c -> "; " ^ c
  | Empty -> ""

let string_of_prog p =
  "(set-logic QF_ABV)\n" ^
    "(set-info :smt-lib-version 2.0)\n" ^
    String.concat "\n" (List.map string_of_stmt p) ^ "\n"

let mkv v = V v
let mkconst c = Const c
let mkconstb c = Const (Bin c)
let mkconsth c = Const (Hex c)
let mkconcat e1 e2 = Concat (e1, e2)
let mkextract e i j = Extract (e, i, j)
let mksll e i = Sll (e, i)
let mksrl e i = Srl (e, i)
let mksra e i = Sra (e, i)
let mksextend e i = SignExtend (e, i)
let mkzextend e i = ZeroExtend (e, i)
let mknot e = Not e
let mkand e1 e2 = And (e1, e2)
let mkor e1 e2 = Or (e1, e2)
let mkxor e1 e2 = Xor (e1, e2)
let mknand e1 e2 = Nand (e1, e2)
let mknor e1 e2 = Nor (e1, e2)
let mkxnor e1 e2 = Xnor (e1, e2)
let mkneg e = Neg e
let mkadd e1 e2 = Add (e1, e2)
let mksum size es =
  match es with
    [] -> mkconstb (zeros size)
  | hd::[] -> hd
  | hd::tl -> List.fold_left (fun res e -> mkadd res e) hd tl
let mksub e1 e2 = Sub (e1, e2)
let mkmul e1 e2 = Mul (e1, e2)
let mksmod e1 e2 = Smod (e1, e2)
let mkite b e1 e2 = Ite (b, e1, e2)
let mkselect e i = Select (e, i)
let mkstore e i v = Store (e, i, v)

let mkeq e1 e2 = Eq (e1, e2)
let mkult e1 e2 = Ult (e1, e2)
let mkule e1 e2 = Ule (e1, e2)
let mkugt e1 e2 = Ugt (e1, e2)
let mkuge e1 e2 = Uge (e1, e2)
let mkslt e1 e2 = Slt (e1, e2)
let mksle e1 e2 = Sle (e1, e2)
let mksgt e1 e2 = Sgt (e1, e2)
let mksge e1 e2 = Sge (e1, e2)
let mkband e1 e2 = Band (e1, e2)
let mkbands es =
  match es with
    [] -> True
  | hd::[] -> hd
  | hd::tl -> List.fold_left (fun res e -> Band (res, e)) hd tl
let mkbor e1 e2 = Bor (e1, e2)
let mkbors es =
  match es with
    [] -> False
  | hd::[] -> hd
  | hd::tl -> List.fold_left (fun res e -> Bor (res, e)) hd tl
let mkbnot e = Bnot e

let mkvar v n = Var (v, n)
let mkarray v n m = Array (v, n, m)
let mkdefine v n e = Define (v, n, e)
let mkassert e = Assert e
let mksat e = Sat e
let mkcomment c = Comment c
let mkempty () = Empty
