
(** A variable in Z3. *)
type var = string

(** Constants. *)
type const =
  Bin of string
| Hex of string

(** An expression in Z3. *)
type exp =
  V of var
| Const of const
| Concat of exp * exp
| Extract of exp * int * int
| Sll of exp * exp
| Srl of exp * exp
| Sra of exp * exp
| SignExtend of exp * int
| ZeroExtend of exp * int
| Not of exp
| And of exp * exp
| Or of exp * exp
| Xor of exp * exp
| Nand of exp * exp
| Nor of exp * exp
| Xnor of exp * exp
| Neg of exp
| Add of exp * exp
| Sub of exp * exp
| Mul of exp * exp
| Smod of exp * exp
| Ite of bexp * exp * exp
| Select of exp * exp
| Store of exp * exp * exp

(** A Boolean expression in Z3. *)
and bexp =
  True
| False
| Eq of exp * exp
| Ult of exp * exp
| Ule of exp * exp
| Ugt of exp * exp
| Uge of exp * exp
| Slt of exp * exp
| Sle of exp * exp
| Sgt of exp * exp
| Sge of exp * exp
| Band of bexp * bexp
| Bor of bexp * bexp
| Bnot of bexp

(** Statements in Z3. *)
type stmt = 
  Var of var * int
| Array of var * int * int
| Define of var * int * exp
| Assert of bexp
| Sat of bexp
| Comment of string
| Empty

(** A program in Z3. *)
type prog = stmt list

val string_of_exp : exp -> string

val string_of_bexp : bexp -> string

val string_of_stmt : stmt -> string

val string_of_prog : prog -> string

val mkv : string -> exp
val mkconst : const -> exp
val mkconstb : string -> exp
val mkconsth : string -> exp
val mkconcat : exp -> exp -> exp
val mkextract : exp -> int -> int -> exp
val mksll : exp -> exp -> exp
val mksrl : exp -> exp -> exp
val mksra : exp -> exp -> exp
val mksextend : exp -> int -> exp
val mkzextend : exp -> int -> exp
val mknot : exp -> exp
val mkand : exp -> exp -> exp
val mkor : exp -> exp -> exp
val mkxor : exp -> exp -> exp
val mknand : exp -> exp -> exp
val mknor : exp -> exp -> exp
val mkxnor : exp -> exp -> exp
val mkneg : exp -> exp
val mkadd : exp -> exp -> exp
val mksum : int -> exp list -> exp
val mksub : exp -> exp -> exp
val mkmul : exp -> exp -> exp
val mksmod : exp -> exp -> exp
val mkite : bexp -> exp -> exp -> exp
val mkselect : exp -> exp -> exp
val mkstore : exp -> exp -> exp -> exp

val mkeq : exp -> exp -> bexp
val mkult : exp -> exp -> bexp
val mkule : exp -> exp -> bexp
val mkugt : exp -> exp -> bexp
val mkuge : exp -> exp -> bexp
val mkslt : exp -> exp -> bexp
val mksle : exp -> exp -> bexp
val mksgt : exp -> exp -> bexp
val mksge : exp -> exp -> bexp
val mkband : bexp -> bexp -> bexp
val mkbands : bexp list -> bexp
val mkbor : bexp -> bexp -> bexp
val mkbors : bexp list -> bexp
val mkbnot : bexp -> bexp

val mkvar : var -> int -> stmt
val mkarray : var -> int -> int -> stmt
val mkdefine : var -> int -> exp -> stmt
val mkassert : bexp -> stmt
val mksat : bexp -> stmt
val mkcomment : string -> stmt
val mkempty : unit -> stmt
