(** A variable in STP. *)
type var = string

(** Constants. *)
type const =
  Bin of string
| Hex of string

(** An expression in STP. *)
type exp =
  V of var
| Const of const
| Concat of exp * exp
| Extract of exp * int * int
| Sll of exp * int
| Srl of exp * int
| Extend of exp * int
| Not of exp
| And of exp * exp
| Or of exp * exp
| Xor of exp * exp
| Nand of exp * exp
| Nor of exp * exp
| Xnor of exp * exp
| Neg of exp
| Add of int * exp * exp
| Sub of int * exp * exp
| Mul of int * exp * exp
| Div of int * exp * exp
| Sdiv of int * exp * exp
| Mod of int * exp * exp
| Smod of int * exp * exp
| Ifte of bexp * exp * exp
| Read of exp * exp
| Write of exp * exp * exp

(** A Boolean expression in STP. *)
and bexp =
  True
| False
| Eq of exp * exp
| Lt of exp * exp
| Le of exp * exp
| Gt of exp * exp
| Ge of exp * exp
| Slt of exp * exp
| Sle of exp * exp
| Sgt of exp * exp
| Sge of exp * exp
| Band of bexp * bexp
| Bor of bexp * bexp
| Bnot of bexp

(** Variable declaration in STP. *)
type decl =
  Var of var * int
| Array of var * int * int

(** Statements in STP. *)
type stmt = 
| Assert of bexp
| Comment of string

(** A program in STP. *)
type prog = {
  vdecls: decl list;
  vstmts: stmt list;
  vquery: bexp
}

val string_of_exp : exp -> string

val string_of_bexp : bexp -> string

val string_of_decl : decl -> string

val string_of_stmt : stmt -> string

val string_of_prog : prog -> string

val mkv : string -> exp
val mkconst : const -> exp
val mkconstb : string -> exp
val mkconsth : string -> exp
val mkconcat : exp -> exp -> exp
val mkextract : exp -> int -> int -> exp
val mksll : exp -> int -> exp
val mksrl : exp -> int -> exp
val mkextend : exp -> int -> exp
val mknot : exp -> exp
val mkand : exp -> exp -> exp
val mkor : exp -> exp -> exp
val mkxor : exp -> exp -> exp
val mknand : exp -> exp -> exp
val mknor : exp -> exp -> exp
val mkxnor : exp -> exp -> exp
val mkneg : exp -> exp
val mkadd : int -> exp -> exp -> exp
val mksum : int -> exp list -> exp
val mksub : int -> exp -> exp -> exp
val mkmul : int -> exp -> exp -> exp
val mkdiv : int -> exp -> exp -> exp
val mksdiv : int -> exp -> exp -> exp
val mkmod : int -> exp -> exp -> exp
val mksmod : int -> exp -> exp -> exp
val mkifte : bexp -> exp -> exp -> exp
val mkread : exp -> exp -> exp
val mkwrite : exp -> exp -> exp -> exp

val mkeq : exp -> exp -> bexp
val mklt : exp -> exp -> bexp
val mkle : exp -> exp -> bexp
val mkgt : exp -> exp -> bexp
val mkge : exp -> exp -> bexp
val mkslt : exp -> exp -> bexp
val mksle : exp -> exp -> bexp
val mksgt : exp -> exp -> bexp
val mksge : exp -> exp -> bexp
val mkband : bexp -> bexp -> bexp
val mkbands : bexp list -> bexp
val mkbor : bexp -> bexp -> bexp
val mkbors : bexp list -> bexp
val mkbnot : bexp -> bexp

val mkvar : var -> int -> decl
val mkarray : var -> int -> int -> decl
val mkassert : bexp -> stmt
val mkcomment : string -> stmt
